
import React, { useState, useMemo } from 'react';
import {
    Truck, Building2, User, Smartphone, StickyNote, Search, Plus,
    Mail, MapPin, Edit3, Trash2, X, Phone, ShieldCheck, BadgeCheck, Hash,
    FileText, Banknote, History, ArrowUpRight, ArrowDownLeft, TrendingUp,
    Filter, Printer, MoreVertical, LayoutGrid, List, Wallet, PieChart
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export function SuppliersPage({ suppliers, purchaseOrders, vendorPayments, notify }: any) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [isLedgerOpen, setIsLedgerOpen] = useState(false);
    const [viewingSupplier, setViewingSupplier] = useState<any>(null);
    const [editingId, setEditingId] = useState<string | null>(null);
    const [searchTerm, setSearchTerm] = useState('');
    const [activeLedgerTab, setActiveLedgerTab] = useState<'orders' | 'payments'>('orders');

    const [formData, setFormData] = useState({
        name: '', person: '', contact: '', email: '', address: '',
        status: 'Active', notes: '', refNumber: ''
    });

    // --- Analytics Engine ---
    const supplierAnalytics = useMemo(() => {
        const stats: any = {};
        let globalPayable = 0;
        let globalVolume = 0;

        suppliers.forEach((s: any) => {
            const supplierPOs = purchaseOrders.filter((po: any) => (po.supplierId || po.supplier_id) === s.id);
            const supplierPayments = vendorPayments.filter((p: any) => p.supplier_id === s.id);

            const totalOrdered = supplierPOs.reduce((acc: number, po: any) => acc + (Number(po.totalAmount || po.total_amount) || 0), 0);
            const totalPaid = supplierPayments.reduce((acc: number, p: any) => acc + (Number(p.amount) || 0), 0);
            const balance = totalOrdered - totalPaid;

            if(balance > 0) globalPayable += balance;
            globalVolume += totalOrdered;

            stats[s.id] = { totalOrdered, totalPaid, balance };
        });

        return { stats, globalPayable, globalVolume };
    }, [suppliers, purchaseOrders, vendorPayments]);

    const filteredSuppliers = useMemo(() => {
        const list = suppliers || [];
        if (!searchTerm) return list;
        const lower = searchTerm.toLowerCase();
        return list.filter((s: any) =>
            s.name.toLowerCase().includes(lower) ||
            (s.contact && s.contact.includes(lower)) ||
            (s.refNumber && s.refNumber.toLowerCase().includes(lower))
        );
    }, [suppliers, searchTerm]);

    // --- Actions ---
    const handleSave = () => {
        if (!formData.name) return;
        const url = editingId ? `/suppliers/${editingId}` : '/suppliers';
        router.post('/suppliers', { ...formData, id: editingId }, {
            onSuccess: () => {
                notify(editingId ? 'Supplier Profile Updated' : 'New Vendor Onboarded');
                closeModal();
            }
        });
    };

    const handleDelete = (id: string, name: string) => {
        if (confirm(`Delete record for ${name}? This will remove linked history reference.`)) {
            router.delete(`/suppliers/${id}`, {
                onSuccess: () => notify('Supplier Removed')
            });
        }
    };

    const openEditModal = (supplier: any) => {
        setEditingId(supplier.id);
        setFormData({ ...supplier, refNumber: supplier.refNumber || '' });
        setIsModalOpen(true);
    };

    const openLedger = (supplier: any) => {
        setViewingSupplier(supplier);
        setIsLedgerOpen(true);
        setActiveLedgerTab('orders');
    };

    const closeModal = () => {
        setIsModalOpen(false);
        setEditingId(null);
        setFormData({ name: '', person: '', contact: '', email: '', address: '', status: 'Active', notes: '', refNumber: '' });
    };

    return (
        <div className="animate-fade-in space-y-8 pb-20">
            {/* KPI Dashboard */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl flex items-center gap-5 relative overflow-hidden group">
                    <div className="absolute right-0 top-0 p-4 opacity-10 group-hover:scale-110 transition-transform"><Wallet size={80}/></div>
                    <div className="w-14 h-14 bg-rose-600 rounded-2xl flex items-center justify-center text-white shadow-lg"><ArrowDownLeft size={28}/></div>
                    <div>
                        <p className="text-[10px] font-black text-rose-300 uppercase tracking-widest mb-1">Total Payable</p>
                        <h3 className="text-3xl font-black font-mono tracking-tight">{formatCurrency(supplierAnalytics.globalPayable)}</h3>
                        <p className="text-[9px] font-bold text-slate-400 uppercase mt-2">Outstanding Liability</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-blue-50 text-blue-600 rounded-xl flex items-center justify-center shadow-inner"><Truck size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Active Vendors</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{suppliers.filter((s:any) => s.status === 'Active').length} <span className="text-xs text-slate-400 font-bold">/ {suppliers.length}</span></p>
                        <p className="text-[8px] font-bold text-blue-500 uppercase">Supply Chain Partners</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-emerald-50 text-emerald-600 rounded-xl flex items-center justify-center shadow-inner"><TrendingUp size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Lifetime Volume</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(supplierAnalytics.globalVolume)}</p>
                        <p className="text-[8px] font-bold text-emerald-500 uppercase">Total Inventory Value</p>
                    </div>
                </div>
            </div>

            {/* Controls */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div>
                    <h3 className="text-lg font-bold font-heading text-slate-900 uppercase italic">Supplier Directory</h3>
                    <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest mt-1">Vendor Relationship Management</p>
                </div>
                <div className="flex gap-2 w-full sm:w-auto">
                    <div className="relative flex-1 sm:w-64">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={14} />
                        <input type="text" placeholder="Search vendors..." className="w-full pl-9 pr-4 py-2.5 bg-white border border-slate-200 rounded-xl text-xs font-bold uppercase tracking-widest outline-none shadow-sm focus:border-indigo-500 transition-all" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                    </div>
                    <button onClick={() => { setEditingId(null); setFormData({ name: '', person: '', contact: '', email: '', address: '', status: 'Active', notes: '', refNumber: '' }); setIsModalOpen(true); }} className="flex items-center gap-2 px-6 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-bold uppercase tracking-widest shadow-lg hover:bg-black transition-all">
                        <Plus size={14} /> Onboard Vendor
                    </button>
                </div>
            </div>

            {/* Vendor Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 px-4 lg:px-0">
                {filteredSuppliers.map((s: any) => {
                    const stat = supplierAnalytics.stats[s.id] || { totalOrdered: 0, totalPaid: 0, balance: 0 };
                    return (
                        <div key={s.id} className="bg-white rounded-[2.5rem] border border-slate-200 p-6 shadow-sm hover:shadow-xl hover:-translate-y-1 transition-all group relative overflow-hidden flex flex-col">
                            {/* Card Header */}
                            <div className="flex justify-between items-start mb-6">
                                <div className="flex items-center gap-3">
                                    <div className="w-12 h-12 bg-slate-50 rounded-2xl flex items-center justify-center text-slate-900 font-bold text-lg shadow-inner">
                                        {s.name.charAt(0)}
                                    </div>
                                    <div>
                                        <h4 className="font-black text-slate-900 uppercase text-sm truncate max-w-[140px]" title={s.name}>{s.name}</h4>
                                        <div className="flex items-center gap-2 mt-1">
                                            <span className={`w-2 h-2 rounded-full ${s.status === 'Active' ? 'bg-emerald-500 animate-pulse' : 'bg-slate-300'}`}></span>
                                            <span className="text-[9px] font-bold text-slate-400 uppercase tracking-wider">{s.status}</span>
                                        </div>
                                    </div>
                                </div>
                                <div className="flex gap-1">
                                    <button onClick={() => openEditModal(s)} className="p-2 hover:bg-slate-50 rounded-lg text-slate-400 hover:text-indigo-600 transition-colors"><Edit3 size={14}/></button>
                                    <button onClick={() => handleDelete(s.id, s.name)} className="p-2 hover:bg-slate-50 rounded-lg text-slate-400 hover:text-rose-600 transition-colors"><Trash2 size={14}/></button>
                                </div>
                            </div>

                            {/* Contact Details */}
                            <div className="space-y-3 mb-6">
                                <div className="flex items-center gap-3 p-2.5 bg-slate-50 rounded-xl border border-slate-100">
                                    <User size={14} className="text-slate-400"/>
                                    <div>
                                        <p className="text-[8px] font-bold text-slate-400 uppercase leading-none mb-0.5">Contact Person</p>
                                        <p className="text-[10px] font-black text-slate-700 uppercase">{s.person || 'N/A'}</p>
                                    </div>
                                </div>
                                <div className="flex items-center gap-3 px-2.5">
                                    <Phone size={14} className="text-slate-300"/>
                                    <span className="text-[10px] font-mono font-bold text-slate-600">{s.contact}</span>
                                </div>
                                {s.address && (
                                    <div className="flex items-start gap-3 px-2.5">
                                        <MapPin size={14} className="text-slate-300 mt-0.5"/>
                                        <span className="text-[9px] font-bold text-slate-500 uppercase leading-relaxed line-clamp-1">{s.address}</span>
                                    </div>
                                )}
                            </div>

                            {/* Financial Mini-Stats */}
                            <div className="mt-auto grid grid-cols-2 gap-3 border-t border-slate-100 pt-4">
                                <div>
                                    <p className="text-[8px] font-black text-slate-400 uppercase tracking-widest mb-1">Liability</p>
                                    <p className={`text-sm font-black font-mono ${stat.balance > 0 ? 'text-rose-600' : 'text-emerald-600'}`}>
                                        {formatCurrency(stat.balance)}
                                    </p>
                                </div>
                                <div className="text-right">
                                    <p className="text-[8px] font-black text-slate-400 uppercase tracking-widest mb-1">Volume</p>
                                    <p className="text-sm font-black font-mono text-slate-900">{formatCurrency(stat.totalOrdered)}</p>
                                </div>
                            </div>

                            {/* Action Footer */}
                            <button onClick={() => openLedger(s)} className="mt-4 w-full py-3 bg-slate-900 text-white rounded-xl text-[10px] font-black uppercase tracking-widest hover:bg-black transition-all flex items-center justify-center gap-2 shadow-lg group-hover:shadow-xl">
                                <History size={12}/> View Full Ledger
                            </button>
                        </div>
                    );
                })}
            </div>

            {/* Create/Edit Modal */}
            {isModalOpen && (
                <div className="fixed inset-0 bg-slate-900/60 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex justify-between items-center border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-slate-900 rounded-2xl flex items-center justify-center text-white"><Building2 size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">{editingId ? 'Edit Vendor' : 'New Vendor'}</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1">Supply Chain Profile</p>
                                </div>
                            </div>
                            <button onClick={closeModal} className="p-2 hover:bg-slate-50 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="space-y-5">
                            <IntegratedInput label="Company / Vendor Name" value={formData.name} onChange={(e: any) => setFormData({...formData, name: e.target.value})} icon={Building2} />

                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Key Contact Person" value={formData.person} onChange={(e: any) => setFormData({...formData, person: e.target.value})} icon={User} />
                                <IntegratedInput label="Phone / Mobile" value={formData.contact} onChange={(e: any) => setFormData({...formData, contact: e.target.value})} icon={Smartphone} />
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Email Address" value={formData.email} onChange={(e: any) => setFormData({...formData, email: e.target.value})} icon={Mail} />
                                <IntegratedInput label="Internal Reference ID" value={formData.refNumber} onChange={(e: any) => setFormData({...formData, refNumber: e.target.value})} icon={Hash} />
                            </div>

                            <IntegratedInput label="Business Address" value={formData.address} onChange={(e: any) => setFormData({...formData, address: e.target.value})} icon={MapPin} />

                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedSelect label="Status" value={formData.status} onChange={(e: any) => setFormData({...formData, status: e.target.value})} options={['Active', 'Inactive']} icon={ShieldCheck} />
                                <IntegratedInput label="Notes" value={formData.notes} onChange={(e: any) => setFormData({...formData, notes: e.target.value})} icon={StickyNote} placeholder="Terms, credit limit..." />
                            </div>
                        </div>

                        <button onClick={handleSave} className="w-full bg-slate-900 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-black transition-all flex items-center justify-center gap-2">
                            <BadgeCheck size={18} /> {editingId ? 'Update Vendor Profile' : 'Register Vendor'}
                        </button>
                    </div>
                </div>
            )}

            {/* Ledger Modal */}
            {isLedgerOpen && viewingSupplier && (
                <div className="fixed inset-0 bg-slate-950/90 z-[110] flex items-center justify-center p-4 backdrop-blur-md overflow-y-auto leading-none">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-5xl shadow-2xl my-auto animate-fade-in relative flex flex-col max-h-[90vh] overflow-hidden border border-white/10">
                        {/* Ledger Header */}
                        <div className="bg-slate-950 p-8 shrink-0 flex justify-between items-center text-white">
                            <div className="flex items-center gap-5">
                                <div className="w-14 h-14 bg-white/10 rounded-2xl flex items-center justify-center text-cyan-400 backdrop-blur-sm border border-white/5"><History size={28} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-lg tracking-widest leading-none">{viewingSupplier.name}</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-2 tracking-[0.2em]">Financial Ledger & History</p>
                                </div>
                            </div>
                            <div className="flex gap-3">
                                <button className="p-3 bg-white/5 hover:bg-white/10 rounded-xl transition-colors text-white" title="Print Ledger"><Printer size={18}/></button>
                                <button onClick={() => { setIsLedgerOpen(false); setViewingSupplier(null); }} className="p-3 bg-rose-500/10 hover:bg-rose-500/20 text-rose-400 rounded-xl transition-colors"><X size={18} /></button>
                            </div>
                        </div>

                        <div className="p-2 bg-slate-100 border-b border-slate-200 flex gap-1 shrink-0">
                            <button onClick={() => setActiveLedgerTab('orders')} className={`flex-1 py-3 rounded-xl text-[10px] font-black uppercase tracking-widest transition-all flex items-center justify-center gap-2 ${activeLedgerTab === 'orders' ? 'bg-white shadow-sm text-slate-900 ring-1 ring-black/5' : 'text-slate-400 hover:bg-white/50'}`}>
                                <FileText size={14} /> Purchase Orders
                            </button>
                            <button onClick={() => setActiveLedgerTab('payments')} className={`flex-1 py-3 rounded-xl text-[10px] font-black uppercase tracking-widest transition-all flex items-center justify-center gap-2 ${activeLedgerTab === 'payments' ? 'bg-white shadow-sm text-slate-900 ring-1 ring-black/5' : 'text-slate-400 hover:bg-white/50'}`}>
                                <Banknote size={14} /> Payment History
                            </button>
                        </div>

                        <div className="flex-1 overflow-y-auto custom-scrollbar p-0 bg-white">
                            {activeLedgerTab === 'orders' ? (
                                <table className="w-full text-left">
                                    <thead className="bg-slate-50 text-[9px] font-black uppercase text-slate-500 sticky top-0 border-b border-slate-100">
                                    <tr><th className="px-8 py-5">Date</th><th className="px-8 py-5">PO Reference</th><th className="px-8 py-5 text-center">Status</th><th className="px-8 py-5 text-right">Total Amount</th></tr>
                                    </thead>
                                    <tbody className="divide-y divide-slate-100">
                                    {purchaseOrders.filter((po:any) => (po.supplierId || po.supplier_id) === viewingSupplier.id).map((po:any) => (
                                        <tr key={po.id} className="hover:bg-slate-50 transition-colors">
                                            <td className="px-8 py-5 text-xs font-bold text-slate-600">{po.date}</td>
                                            <td className="px-8 py-5 text-xs font-bold text-slate-900 font-mono uppercase">{po.refNumber || po.ref_number || '---'}</td>
                                            <td className="px-8 py-5 text-center"><span className={`px-3 py-1 rounded-full text-[8px] font-black uppercase border ${po.status === 'Received' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' : 'bg-amber-50 text-amber-600 border-amber-100'}`}>{po.status || 'Pending'}</span></td>
                                            <td className="px-8 py-5 text-right font-black font-mono text-slate-900">{formatCurrency(po.totalAmount || po.total_amount)}</td>
                                        </tr>
                                    ))}
                                    {purchaseOrders.filter((po:any) => (po.supplierId || po.supplier_id) === viewingSupplier.id).length === 0 && (
                                        <tr><td colSpan={4} className="p-16 text-center text-slate-400 text-[10px] font-bold uppercase italic tracking-widest">No Purchase Orders Recorded</td></tr>
                                    )}
                                    </tbody>
                                </table>
                            ) : (
                                <table className="w-full text-left">
                                    <thead className="bg-slate-50 text-[9px] font-black uppercase text-slate-500 sticky top-0 border-b border-slate-100">
                                    <tr><th className="px-8 py-5">Payment Date</th><th className="px-8 py-5">Reference / Challan</th><th className="px-8 py-5">Notes</th><th className="px-8 py-5 text-right">Amount Paid</th></tr>
                                    </thead>
                                    <tbody className="divide-y divide-slate-100">
                                    {vendorPayments.filter((p:any) => p.supplier_id === viewingSupplier.id).map((p:any) => (
                                        <tr key={p.id} className="hover:bg-slate-50 transition-colors">
                                            <td className="px-8 py-5 text-xs font-bold text-slate-600">{p.payment_date}</td>
                                            <td className="px-8 py-5 text-xs font-bold text-slate-900 font-mono uppercase">{p.ref_number || '---'}</td>
                                            <td className="px-8 py-5 text-[10px] font-bold text-slate-500 uppercase max-w-xs truncate">{p.notes}</td>
                                            <td className="px-8 py-5 text-right font-black font-mono text-emerald-600">{formatCurrency(p.amount)}</td>
                                        </tr>
                                    ))}
                                    {vendorPayments.filter((p:any) => p.supplier_id === viewingSupplier.id).length === 0 && (
                                        <tr><td colSpan={4} className="p-16 text-center text-slate-400 text-[10px] font-bold uppercase italic tracking-widest">No Payments Recorded</td></tr>
                                    )}
                                    </tbody>
                                </table>
                            )}
                        </div>

                        {/* Ledger Footer Summary */}
                        <div className="p-8 bg-slate-50 border-t border-slate-200 shrink-0 flex justify-between items-center">
                            <div>
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Net Payable Balance</p>
                                <p className={`text-3xl font-black font-mono tracking-tighter ${supplierAnalytics.stats[viewingSupplier.id]?.balance > 0 ? 'text-rose-600' : 'text-emerald-600'}`}>
                                    {formatCurrency(supplierAnalytics.stats[viewingSupplier.id]?.balance || 0)}
                                </p>
                            </div>
                            <div className="flex gap-8 text-right">
                                <div>
                                    <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Purchases</p>
                                    <p className="text-lg font-black font-mono text-slate-900">{formatCurrency(supplierAnalytics.stats[viewingSupplier.id]?.totalOrdered || 0)}</p>
                                </div>
                                <div>
                                    <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Paid</p>
                                    <p className="text-lg font-black font-mono text-emerald-600">{formatCurrency(supplierAnalytics.stats[viewingSupplier.id]?.totalPaid || 0)}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
