

import React, { useState, useMemo, useEffect } from 'react';
import {
    Printer, FileSpreadsheet, TrendingUp, TrendingDown, Box,
    Wallet, Receipt, Download, FileText, ChevronRight,
    X, Search, Filter, ArrowLeft, BadgeCheck, DollarSign,
    Briefcase, Activity, Landmark, History, Bike, Calendar,
    RefreshCw, Users, User, Scale, Calculator, ChartNoAxesCombined as ChartIcon,
    Fingerprint, MapPin, Truck, ShieldAlert, Gavel, ShoppingBag,
    ArrowRight, ClipboardCheck, Info, Clock, CalendarCheck, AlertTriangle,
    HandCoins, BookOpen, Languages, Scissors, QrCode, Smartphone, CheckCircle2,
    Banknote, CreditCard, PieChart, BarChart4, Layers, AlertOctagon,
    ArrowUpRight, ArrowDownRight, Minus, Percent, Plus, Hourglass, FileCheck,
    Building2, Factory, FileBadge
} from 'lucide-react';
import { formatCurrency, getCurrentDate, IntegratedInput, IntegratedSelect } from './CommonUI.tsx';

type ReportType = 'sales' | 'stock' | 'expenses' | 'financials' | 'customer_ledger' | 'supplier_ledger' | 'balance_sheet' | 'trial_balance' | 'bike_trace' | 'pending_po' | 'purchases' | 'bookings' | 'daily_roznamcha' | 'urdu_roznamcha' | 'sale_invoice' | 'accounts_summary' | 'monthly_summary' | null;

// Urdu Translation Dictionary
const URDU_DICT: any = {
    'Inflow': 'آمدنی',
    'Outflow': 'اخراجات',
    'Sale Recovery': 'بقایا وصولی',
    'Sale Receipt': 'فروخت وصولی',
    'New Sale': 'نئی فروخت',
    'Operational Expense': 'دفتر خرچہ',
    'Stock Purchase': 'خریداری مال (ادائیگی)',
    'Capital Investment': 'سرمایہ کاری',
    'Capital Withdrawal': 'مالک کا خرچہ',
    'Description': 'تفصیل',
    'Type': 'قسم',
    'Amount': 'رقم',
    'Date': 'تاریخ',
    'Daily Roznamcha': 'روزنامچہ',
    'Showroom Name': 'شوروم کا نام',
    'Account': 'اکاؤنٹ',
    'Ref': 'حوالہ',
    'Opening Balance': 'ابتدائی بیلنس',
    'Closing Balance': 'اختتامی بیلنس',
    'Fee Recovery': 'کمپنی کلیم وصولی',
    'Registration Profit': 'رجسٹریشن منافع',
    'Registration Fee': 'رجسٹریشن وصولی',
    'Registration Agent Fee': 'رجسٹریشن خرچہ (ایجنٹ)',
    'Booking Advance': 'بکنگ ایڈوانس',
    'Workshop Service': 'ورکشاپ خرچہ',
    'Warranty Recovery': 'وارنٹی وصولی',
    'Stock In': 'آمد مال (اسٹاک)',
    'Inventory Received': 'اسٹاک وصولی'
};

// Helper to safely extract numbers from snake_case or camelCase
const getVal = (obj: any, keys: string[]) => {
    if (!obj) return 0;
    for (const key of keys) {
        if (obj[key] !== undefined && obj[key] !== null) {
            return Number(obj[key]) || 0;
        }
    }
    return 0;
};

// Helper to safely extract strings
const getStr = (obj: any, keys: string[]) => {
    if (!obj) return '';
    for (const key of keys) {
        if (obj[key] !== undefined && obj[key] !== null) {
            return String(obj[key]);
        }
    }
    return '';
};

// Date Range Helper
const isWithinRange = (dateStr: string, from: string, to: string) => {
    if (!dateStr) return false;
    const d = dateStr.substring(0, 10);
    return d >= from && d <= to;
};

// Aging Helper
const getDaysDifference = (dateStr: string) => {
    if (!dateStr) return 0;
    const start = new Date(dateStr);
    const end = new Date();
    if (isNaN(start.getTime())) return 0;
    const diffTime = Math.abs(end.getTime() - start.getTime());
    return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
};

export function ReportsPage({
                                salesHistory, physicalStock, expenses, products, feeBatches,
                                registrations, accounts, customers, capital, transferHistory,
                                claims, dealerExchanges, bookings, showroomSettings, purchaseOrders, suppliers,
                                vendorPayments,
                                requestedReport, setRequestedReport
                            }: any) {
    const [activeReport, setActiveReport] = useState<ReportType>(null);
    const [isPrintModalOpen, setIsPrintModalOpen] = useState(false);
    const [searchTerm, setSearchTerm] = useState('');
    const [selectedCustomerId, setSelectedCustomerId] = useState('');
    const [selectedSupplierId, setSelectedSupplierId] = useState('');
    const [traceInput, setTraceInput] = useState('');
    const [filterYear, setFilterYear] = useState(new Date().getFullYear().toString());
    const [monthFrom, setMonthFrom] = useState('01');
    const [monthTo, setMonthTo] = useState('12');

    // --- Invoice Archive Specific State ---
    const [viewingInvoice, setViewingInvoice] = useState<any>(null); // For Invoice Preview Modal

    // --- Universal Period Filters ---
    const [dateFrom, setDateFrom] = useState(getCurrentDate());
    const [dateTo, setDateTo] = useState(getCurrentDate());

    // Auto-load requested report if sent from another module
    useEffect(() => {
        if (requestedReport) {
            setActiveReport(requestedReport as ReportType);
            setRequestedReport(null); // Clear after use
        }
    }, [requestedReport, setRequestedReport]);

    const reportCategories = [
        { id: 'daily_roznamcha', title: 'Daily Report (Roznamcha)', desc: 'Consolidated daily activity & cash flow.', icon: BookOpen, color: 'bg-emerald-50 text-emerald-600' },
        { id: 'urdu_roznamcha', title: 'Urdu Roznamcha (روزنامچہ)', desc: 'Daily data in RTL Urdu format.', icon: Languages, color: 'bg-blue-50 text-blue-600' },
        { id: 'financials', title: 'Executive P&L Statement', desc: 'Net profit, COGS, and margin analysis.', icon: PieChart, color: 'bg-indigo-50 text-indigo-600' },
        { id: 'sale_invoice', title: 'Sale Invoice Archive', desc: 'Search and regenerate professional A4 invoices.', icon: FileText, color: 'bg-slate-900 text-white' },
        { id: 'stock', title: 'Inventory Audit & Aging', desc: 'Valuation, physical count, and aging.', icon: Box, color: 'bg-cyan-50 text-cyan-600' },
        { id: 'monthly_summary', title: 'Monthly Performance Ledger', desc: 'Aggregated profit & expense trends by month.', icon: BarChart4, color: 'bg-emerald-50 text-emerald-600' },
        { id: 'customer_ledger', title: 'Customer Ledger', desc: 'Complete payment history per client.', icon: Users, color: 'bg-slate-50 text-slate-600' },
        { id: 'supplier_ledger', title: 'Supplier Ledger', desc: 'Payable history and purchasing log.', icon: Truck, color: 'bg-orange-50 text-orange-600' },
        { id: 'sales', title: 'Sales Ledger', desc: 'Invoices, margins, and collection status.', icon: TrendingUp, color: 'bg-slate-50 text-slate-600' },
        { id: 'expenses', title: 'Expense Analysis', desc: 'Operational outflow and utility logs.', icon: Wallet, color: 'bg-rose-50 text-rose-600' },
        { id: 'bike_trace', title: 'Bike DNA Trace', desc: 'Full lifecycle audit of a specific unit.', icon: Fingerprint, color: 'bg-slate-900 text-white' },
        { id: 'accounts_summary', title: 'Liquidity & Accounts Audit', desc: 'Snapshot of all cash, bank and digital balances.', icon: Landmark, color: 'bg-cyan-50 text-cyan-600' },
        { id: 'purchases', title: 'Purchase Journal', desc: 'Historical record of stock acquisitions.', icon: Truck, color: 'bg-blue-50 text-blue-600' },
        { id: 'balance_sheet', title: 'Balance Sheet', desc: 'Assets, Equity, and Liability snapshot.', icon: Scale, color: 'bg-slate-50 text-slate-600' },
    ];

    // Helper for Supplier PO aggregation (Moved before usage)
    const getPoItemsCount = (po: any) => {
        const items = po.items || po.purchase_order_items || [];
        return items.length;
    };

    // --- Standard Filtering Logic ---
    const currentReportData = useMemo(() => {
        if (!activeReport || activeReport === 'bike_trace') return [];
        const term = searchTerm.toLowerCase();
        const from = dateFrom || '1900-01-01';
        const to = dateTo || '2100-01-01';

        const filterByDate = (arr: any[], dateField = 'date') =>
            arr.filter(item => {
                const d = item[dateField] || item['created_at']?.substring(0,10) || '2000-01-01';
                return d >= from && d <= to;
            });

        let data: any[] = [];

        switch (activeReport) {
            case 'daily_roznamcha':
            case 'urdu_roznamcha':
                // ... (Existing Roznamcha Logic) ...
                const rSales = salesHistory.flatMap((s: any) =>
                    (s.payments || []).filter((p: any) => isWithinRange(p.date, from, to)).map((p: any) => ({
                        date: p.date, type: 'Inflow', category: 'Sale Receipt', desc: `${getStr(s, ['customer', 'customerName', 'customer_name'])} - ${s.bike || s.product || 'Unit'}`, amount: getVal(p, ['amount']), ref: p.refNumber || `RCPT-${p.id || '0'}`, mode: 'Cash'
                    }))
                );
                const rExps = expenses.filter((e: any) => isWithinRange(e.date, from, to)).map((e: any) => ({ date: e.date, type: 'Outflow', category: 'Operational Expense', desc: e.description || e.category, amount: getVal(e, ['amount']), ref: e.refNumber || e.id }));
                const rVendorPays = (vendorPayments || []).filter((v: any) => isWithinRange(v.payment_date, from, to)).map((v: any) => { const isWorkshop = v.workshop_id ? true : false; return { date: v.payment_date, type: 'Outflow', category: isWorkshop ? 'Workshop Service' : 'Stock Purchase', desc: v.notes || (isWorkshop ? 'Repair Cost' : 'Supplier Payment'), amount: getVal(v, ['amount']), ref: v.ref_number || `VP-${v.id}` }; });
                const rCaps = capital.filter((c: any) => isWithinRange(c.date, from, to)).map((c: any) => ({ date: c.date, type: c.type === 'Investment' ? 'Inflow' : 'Outflow', category: c.type === 'Investment' ? 'Capital Investment' : 'Capital Withdrawal', desc: c.description, amount: getVal(c, ['amount']), ref: c.refNumber || c.id }));
                const rBoks = bookings.filter((b: any) => isWithinRange(b.date || b.booking_date, from, to)).map((b: any) => ({ date: b.date || b.booking_date, type: 'Inflow', category: 'Booking Advance', desc: `Reservation: ${getStr(b, ['customerName', 'customer_name'])}`, amount: getVal(b, ['advance', 'advance_amount']), ref: b.refNumber || b.id }));
                const rRegsIn = registrations.filter((r: any) => isWithinRange(r.registrationDate || r.registration_date, from, to) && getVal(r, ['customerFee', 'customer_fee']) > 0).map((r: any) => ({ date: r.registrationDate || r.registration_date, type: 'Inflow', category: 'Registration Fee', desc: `Reg Fee: ${getStr(r, ['regNumber', 'reg_number', 'customerName', 'customer_name'])}`, amount: getVal(r, ['customerFee', 'customer_fee']), ref: r.refNumber || r.id }));
                const rRegsOut = registrations.filter((r: any) => isWithinRange(r.registrationDate || r.registration_date, from, to) && getVal(r, ['agentFee', 'agent_fee']) > 0).map((r: any) => ({ date: r.registrationDate || r.registration_date, type: 'Outflow', category: 'Registration Agent Fee', desc: `Agent Payment: ${getStr(r, ['agentName', 'agent_name'])}`, amount: getVal(r, ['agentFee', 'agent_fee']), ref: r.refNumber || r.id }));
                const rFees = (feeBatches || []).flatMap((b:any) => { const recAmount = getVal(b, ['receivedAmount', 'received_amount']); const date = b.updated_at ? b.updated_at.substring(0,10) : (b.date || b.dispatch_date); return isWithinRange(date, from, to) && recAmount > 0 ? [{ date: date, type: 'Inflow', category: 'Fee Recovery', desc: `Claim Reimbursement`, amount: recAmount, ref: b.id }] : [] });
                const rClaimsRec = (claims || []).filter((c: any) => { const status = c.supplierStatus || c.supplier_status; const amt = getVal(c, ['claimedAmount', 'claimed_amount']); const d = c.updated_at ? c.updated_at.substring(0,10) : (c.claimDate || c.claim_date); return status === 'Recovered' && amt > 0 && isWithinRange(d, from, to); }).map((c: any) => ({ date: c.updated_at ? c.updated_at.substring(0,10) : (c.claimDate || c.claim_date), type: 'Inflow', category: 'Warranty Recovery', desc: `Supplier Claim: ${c.engineNumber || c.engine_number}`, amount: getVal(c, ['claimedAmount', 'claimed_amount']), ref: `CLM-${c.id}` }));
                const rStockIn = physicalStock.filter((s: any) => isWithinRange(s.date || s.entry_date, from, to)).map((s: any) => { const pId = s.productId || s.product_id; const prod = products.find((p:any) => String(p.id) === String(pId)); const modelName = prod ? `${prod.make} ${prod.model}` : 'Unknown Unit'; return { date: s.date || s.entry_date, type: 'Stock In', category: 'Inventory Received', desc: `${modelName} - ${s.engineNumber || s.engine_number}`, amount: getVal(s, ['purchasePrice', 'purchase_price']), ref: s.refNumber || `STK-${s.id}` }; });
                data = [...rSales, ...rExps, ...rVendorPays, ...rCaps, ...rBoks, ...rRegsIn, ...rRegsOut, ...rFees, ...rClaimsRec, ...rStockIn];
                data.sort((a: any, b: any) => new Date(a.date).getTime() - new Date(b.date).getTime());
                break;

            case 'monthly_summary':
                // ... (Monthly Summary Logic) ...
                const monthlyMap: any = {};
                const processArr = (arr: any[], dateFields: string[], logic: (m: any, item: any) => void) => { if (!arr) return; arr.forEach(item => { const dateVal = getStr(item, dateFields) || item['created_at']?.substring(0,10); if(!dateVal) return; const year = dateVal.substring(0, 4); const monthPart = dateVal.substring(5, 7); if (filterYear !== 'All' && year !== filterYear) return; if (monthPart < monthFrom || monthPart > monthTo) return; const monthKey = dateVal.substring(0, 7); if (!monthlyMap[monthKey]) monthlyMap[monthKey] = { month: monthKey, sales: 0, cogs: 0, expenses: 0, fees: 0, regProfit: 0 }; logic(monthlyMap[monthKey], item); }); };
                processArr(salesHistory, ['date', 'sale_date'], (m, s) => { m.sales += getVal(s, ['total', 'total_amount']); m.cogs += getVal(s, ['costPrice', 'cost_price', 'purchase_price']); });
                processArr(expenses, ['date', 'expense_date'], (m, e) => { m.expenses += getVal(e, ['amount']); });
                processArr(feeBatches, ['date', 'dispatch_date'], (m, f) => { m.fees += getVal(f, ['receivedAmount', 'received_amount']); });
                processArr(registrations, ['registrationDate', 'registration_date'], (m, r) => { m.regProfit += getVal(r, ['profit']); });
                data = Object.values(monthlyMap).sort((a: any, b: any) => b.month.localeCompare(a.month));
                break;

            case 'sale_invoice':
                // ... (Sale Invoice Logic) ...
                data = filterByDate(salesHistory, 'date').map((s: any) => { const total = getVal(s, ['total', 'total_amount']); const paid = (s.payments || []).reduce((acc: number, p: any) => acc + getVal(p, ['amount']), 0); const balance = total - paid; let customerName = 'Unknown Client'; let customerPhone = '---'; if (s.customerName || s.customer_name) customerName = s.customerName || s.customer_name; else if (typeof s.customer === 'object' && s.customer !== null) { customerName = s.customer.name; customerPhone = s.customer.contact || customerPhone; } else if (s.customerId || s.customer_id) { const cust = customers.find((c:any) => String(c.id) === String(s.customerId || s.customer_id)); if(cust) { customerName = cust.name; customerPhone = cust.contact || customerPhone; } } let productName = 'Vehicle'; let engine = getStr(s, ['engine', 'engineNumber', 'engine_number']) || '---'; let chassis = getStr(s, ['chassis', 'chassisNumber', 'chassis_number']) || '---'; if (s.bike || s.bike_info) productName = s.bike || s.bike_info; else if (s.productId || s.product_id) { const prod = products.find((p:any) => String(p.id) === String(s.productId || s.product_id)); if(prod) productName = `${prod.make} ${prod.model} ${prod.year || ''}`; } return { ...s, id: s.id, total, paid, balance, customerName, customerPhone, productName, engine, chassis, status: balance <= 0 ? 'Settled' : 'Outstanding', paymentMode: s.paymentType || 'Full Payment', date: s.date }; }); if (searchTerm) { const term = searchTerm.toLowerCase(); data = data.filter(s => String(s.customerName).toLowerCase().includes(term) || String(s.id).toLowerCase().includes(term) || String(s.engine).toLowerCase().includes(term) || String(s.chassis).toLowerCase().includes(term)); } data.sort((a: any, b: any) => new Date(b.date).getTime() - new Date(a.date).getTime());
                break;

            case 'stock':
                // ... (Stock Logic) ...
                data = physicalStock.map((s: any) => { const p = products.find((pr: any) => pr.id === (s.productId || s.product_id)); const entryDate = s.date || s.entry_date || getCurrentDate(); const age = getDaysDifference(entryDate); let agingStatus = 'Fresh'; if (age > 90) agingStatus = 'Stagnant'; else if (age > 30) agingStatus = 'Stable'; return { ...s, make: p?.make || 'Unknown', model: p?.model || 'Unit', color: p?.color || '', purchasePrice: getVal(s, ['purchasePrice', 'purchase_price']), age, agingStatus, entryDate }; });
                data.sort((a: any, b: any) => b.age - a.age);
                break;

            case 'customer_ledger':
                if (!selectedCustomerId) return [];
                const clientSales = salesHistory.filter((s: any) => String(s.customerId || s.customer_id) === String(selectedCustomerId));
                const clientPayments = clientSales.flatMap((s: any) => (s.payments || []).map((p: any) => ({ ...p, type: 'Payment', ref: `INV: ${s.id}`, date: p.date, amount: getVal(p, ['amount']), bike: s.bike || 'Vehicle' })));
                const clientInvoices = clientSales.map((s: any) => ({ ...s, type: 'Invoice', amount: getVal(s, ['total', 'total_amount']), ref: `INV: ${s.id}`, date: s.date || s.sale_date, bike: s.bike || s.productName }));
                data = [...clientInvoices, ...clientPayments].sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime());
                break;

            case 'supplier_ledger':
                if (!selectedSupplierId) return [];
                const supPOs = purchaseOrders.filter((p: any) => String(p.supplierId || p.supplier_id) === String(selectedSupplierId));
                const supPays = vendorPayments.filter((p: any) => String(p.supplier_id) === String(selectedSupplierId));

                const ledgerItems = [
                    ...supPOs.map((po: any) => ({
                        date: po.date,
                        type: 'Bill',
                        ref: po.refNumber || `PO-${po.id.toString().substring(0,6)}`,
                        desc: `Stock Purchase: ${getPoItemsCount(po)} Units`,
                        billAmount: getVal(po, ['totalAmount', 'total_amount']),
                        paidAmount: 0
                    })),
                    ...supPays.map((pay: any) => ({
                        date: pay.payment_date,
                        type: 'Payment',
                        ref: pay.ref_number || `VP-${pay.id}`,
                        desc: pay.notes || 'Vendor Payment',
                        billAmount: 0,
                        paidAmount: getVal(pay, ['amount'])
                    }))
                ];
                data = ledgerItems.sort((a: any, b: any) => new Date(a.date).getTime() - new Date(b.date).getTime());
                break;

            case 'sales':
                data = filterByDate(salesHistory, 'date').map((s: any) => {
                    const total = getVal(s, ['total', 'total_amount']);
                    const cost = getVal(s, ['costPrice', 'cost_price', 'purchase_price']);
                    const paid = (s.payments || []).reduce((acc: number, p: any) => acc + getVal(p, ['amount']), 0);
                    return {
                        ...s,
                        total,
                        cost,
                        profit: total - cost,
                        margin: total > 0 ? ((total - cost) / total) * 100 : 0,
                        totalPaid: paid,
                        balance: total - paid
                    };
                });
                if (searchTerm) {
                    const term = searchTerm.toLowerCase();
                    data = data.filter(s => String(s.customerName || s.customer).toLowerCase().includes(term) || String(s.id).includes(term));
                }
                break;

            case 'expenses':
                data = filterByDate(expenses, 'date').map(e => ({
                    ...e,
                    amount: getVal(e, ['amount']),
                    category: e.category || 'General',
                    date: e.date
                })).sort((a:any, b:any) => new Date(b.date).getTime() - new Date(a.date).getTime());
                break;

            // ... (Other cases generic) ...
            case 'financials':
            case 'balance_sheet':
            case 'trial_balance':
            case 'accounts_summary':
            case 'purchases':
                // Reusing previous logic logic block for financials/etc inside the render directly or pre-calc if needed
                // For simplicity, Financials is handled in render logic below as it's complex aggregation
                break;
        }

        return data;
    }, [activeReport, salesHistory, physicalStock, expenses, searchTerm, products, feeBatches, registrations, dateFrom, dateTo, selectedCustomerId, selectedSupplierId, accounts, capital, bookings, filterYear, monthFrom, monthTo, vendorPayments, claims]);

    // --- Report Specific Totals ---
    const reportTotals = useMemo(() => {
        if (!activeReport || activeReport === 'bike_trace') return null;
        if (activeReport === 'daily_roznamcha' || activeReport === 'urdu_roznamcha') { const inflow = currentReportData.filter((d:any) => d.type === 'Inflow').reduce((a:any, b:any) => a + b.amount, 0); const outflow = currentReportData.filter((d:any) => d.type === 'Outflow').reduce((a:any, b:any) => a + b.amount, 0); return { inflow, outflow, closing: inflow - outflow }; }
        if (activeReport === 'monthly_summary') { const rev = currentReportData.reduce((a:any, b:any) => a + b.sales + b.fees + b.regProfit, 0); const exp = currentReportData.reduce((a:any, b:any) => a + b.expenses + b.cogs, 0); return { rev, exp, net: rev - exp }; }
        if (activeReport === 'stock') { const count = currentReportData.length; const value = currentReportData.reduce((acc: number, s: any) => acc + s.purchasePrice, 0); const stagnantValue = currentReportData.filter((s:any) => s.agingStatus === 'Stagnant').reduce((acc: number, s: any) => acc + s.purchasePrice, 0); const freshValue = currentReportData.filter((s:any) => s.agingStatus === 'Fresh').reduce((acc: number, s: any) => acc + s.purchasePrice, 0); return { count, value, stagnantValue, freshValue }; }
        if (activeReport === 'sale_invoice') { const totalInvoiced = currentReportData.reduce((a:any, b:any) => a + b.total, 0); const totalCollected = currentReportData.reduce((a:any, b:any) => a + b.paid, 0); const totalReceivable = currentReportData.reduce((a:any, b:any) => a + b.balance, 0); return { totalInvoiced, totalCollected, totalReceivable, count: currentReportData.length }; }
        if (activeReport === 'financials') return null;
        if (activeReport === 'sales') {
            const rev = currentReportData.reduce((a:any, b:any) => a + b.total, 0);
            const profit = currentReportData.reduce((a:any, b:any) => a + b.profit, 0);
            const cost = currentReportData.reduce((a:any, b:any) => a + b.cost, 0);
            return { rev, profit, cost, count: currentReportData.length };
        }
        if (activeReport === 'expenses') {
            const total = currentReportData.reduce((a:any, b:any) => a + b.amount, 0);
            const byCat = currentReportData.reduce((acc:any, curr:any) => {
                acc[curr.category] = (acc[curr.category] || 0) + curr.amount;
                return acc;
            }, {});
            // Convert to array for chart
            const breakdown = Object.entries(byCat).map(([label, value]) => ({ label, value })).sort((a:any, b:any) => b.value - a.value);
            return { total, breakdown };
        }
        if (activeReport === 'purchases') return { total: currentReportData.reduce((a:any, b:any) => a + b.amount, 0) };
        return null;
    }, [currentReportData, activeReport, dateFrom, salesHistory, expenses, capital, bookings, registrations, vendorPayments]);

    const handleQuickMonth = () => { const now = new Date(); const firstDay = new Date(now.getFullYear(), now.getMonth(), 1).toISOString().split('T')[0]; const lastDay = new Date(now.getFullYear(), now.getMonth() + 1, 0).toISOString().split('T')[0]; setDateFrom(firstDay); setDateTo(lastDay); };
    const handleClearFilters = () => { setDateFrom(getCurrentDate()); setDateTo(getCurrentDate()); setSearchTerm(''); setSelectedCustomerId(''); setSelectedSupplierId(''); setTraceInput(''); setFilterYear(new Date().getFullYear().toString()); setMonthFrom('01'); setMonthTo('12'); };
    const getUrdu = (key: string) => URDU_DICT[key] || key;
    const isUrdu = activeReport === 'urdu_roznamcha';

    return (
        <div className={`animate-fade-in leading-none ${isUrdu ? 'rtl' : ''}`}>

            {/* Screen UI Wrapper - Hides all screen content during print */}
            <div className="space-y-8 pb-20 no-print">
                {/* Header */}
                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 px-4 lg:px-0 leading-none">
                    <div className="flex items-center gap-4">
                        {activeReport && (
                            <button onClick={() => { setActiveReport(null); handleClearFilters(); }} className="p-2.5 bg-white border border-slate-200 rounded-xl text-slate-400 hover:text-slate-900 transition-all shadow-sm">
                                <ArrowLeft size={18} className={isUrdu ? 'rotate-180' : ''} />
                            </button>
                        )}
                        <div>
                            <h3 className={`text-lg font-bold font-heading text-slate-900 uppercase italic leading-none ${isUrdu ? 'urdu-font' : ''}`}>
                                {activeReport ? (isUrdu ? getUrdu('Daily Roznamcha') : reportCategories.find(c => c.id === activeReport)?.title) : 'Financial Terminal'}
                            </h3>
                            <p className={`text-[10px] text-slate-400 font-bold uppercase tracking-widest mt-1.5 leading-none ${isUrdu ? 'urdu-font' : ''}`}>
                                {activeReport ? 'Authorized Internal Audit' : 'Central Management Information System'}
                            </p>
                        </div>
                    </div>
                    {activeReport && (
                        <div className="flex gap-2 w-full sm:w-auto">
                            {activeReport !== 'bike_trace' && activeReport !== 'sale_invoice' && (
                                <button onClick={() => setIsPrintModalOpen(true)} className="px-6 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-bold uppercase tracking-widest shadow-lg hover:bg-black transition-all flex items-center gap-2">
                                    <Printer size={14} /> {isUrdu ? 'پرنٹ دیکھیں' : 'View Terminal'}
                                </button>
                            )}
                        </div>
                    )}
                </div>

                {/* Dynamic Filter Bar */}
                {activeReport && activeReport !== 'monthly_summary' && activeReport !== 'stock' && activeReport !== 'sale_invoice' && (
                    <div className="bg-white p-2 rounded-xl border border-slate-200 flex flex-col sm:flex-row gap-2 mx-4 lg:mx-0 shadow-sm">
                        {activeReport === 'customer_ledger' ? (
                            <div className="flex-1">
                                <IntegratedSelect
                                    label="Select Client Account"
                                    value={selectedCustomerId}
                                    onChange={(e: any) => setSelectedCustomerId(e.target.value)}
                                    options={customers}
                                    displayKey="name"
                                    icon={Users}
                                    placeholder="Search by name..."
                                />
                            </div>
                        ) : activeReport === 'supplier_ledger' ? (
                            <div className="flex-1">
                                <IntegratedSelect
                                    label="Select Vendor Account"
                                    value={selectedSupplierId}
                                    onChange={(e: any) => setSelectedSupplierId(e.target.value)}
                                    options={suppliers}
                                    displayKey="name"
                                    icon={Truck}
                                    placeholder="Search by name..."
                                />
                            </div>
                        ) : activeReport === 'bike_trace' ? (
                            <div className="flex-1 flex items-center gap-2 px-3 bg-slate-50 border border-slate-200 rounded-lg">
                                <Search size={16} className="text-slate-400"/>
                                <input
                                    type="text"
                                    className="w-full py-3 bg-transparent text-xs font-black uppercase outline-none placeholder:text-slate-400"
                                    placeholder="Enter Engine or Chassis Number..."
                                    value={traceInput}
                                    onChange={(e) => setTraceInput(e.target.value)}
                                />
                            </div>
                        ) : activeReport === 'sales' || activeReport === 'expenses' ? (
                            <>
                                <IntegratedInput type="date" value={dateFrom} onChange={(e: any) => setDateFrom(e.target.value)} icon={Calendar} colSpan="flex-1" />
                                <IntegratedInput type="date" value={dateTo} onChange={(e: any) => setDateTo(e.target.value)} icon={Calendar} colSpan="flex-1" />
                                <button onClick={handleQuickMonth} className="px-4 py-2 bg-slate-50 text-slate-500 rounded-lg text-[9px] font-black uppercase tracking-widest hover:bg-slate-100">This Month</button>
                                <div className="relative flex-1">
                                    <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={14} />
                                    <input type="text" placeholder="Search records..." className="w-full pl-9 pr-4 py-2 bg-slate-50 rounded-lg text-[10px] font-black uppercase tracking-widest outline-none" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                                </div>
                            </>
                        ) : (
                            <>
                                <IntegratedInput type="date" value={dateFrom} onChange={(e: any) => setDateFrom(e.target.value)} icon={Calendar} colSpan="flex-1" />
                                {activeReport !== 'daily_roznamcha' && activeReport !== 'urdu_roznamcha' && (
                                    <IntegratedInput type="date" value={dateTo} onChange={(e: any) => setDateTo(e.target.value)} icon={Calendar} colSpan="flex-1" />
                                )}
                                <button onClick={handleQuickMonth} className="px-4 py-2 bg-slate-50 text-slate-500 rounded-lg text-[9px] font-black uppercase tracking-widest hover:bg-slate-100">This Month</button>
                            </>
                        )}
                    </div>
                )}

                {/* Specialized Filters for Invoice & Stock */}
                {(activeReport === 'stock' || activeReport === 'sale_invoice') && (
                    <div className="bg-white p-2 rounded-xl border border-slate-200 flex flex-col sm:flex-row gap-2 mx-4 lg:mx-0 shadow-sm">
                        <div className="flex-1 relative">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={14} />
                            <input
                                type="text"
                                placeholder={activeReport === 'stock' ? "Search Engine / Chassis / Model..." : "Search Invoice ID, Customer, or Unit..."}
                                className="w-full pl-9 pr-4 py-3 bg-slate-50 rounded-lg text-[10px] font-black uppercase tracking-widest outline-none focus:bg-white transition-all"
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                            />
                        </div>
                        {/* Invoice Date Range */}
                        {activeReport === 'sale_invoice' && (
                            <>
                                <IntegratedInput type="date" value={dateFrom} onChange={(e: any) => setDateFrom(e.target.value)} icon={Calendar} />
                                <IntegratedInput type="date" value={dateTo} onChange={(e: any) => setDateTo(e.target.value)} icon={Calendar} />
                            </>
                        )}
                    </div>
                )}

                {/* Report Content */}
                {activeReport === 'daily_roznamcha' || activeReport === 'urdu_roznamcha' ? (
                    /* ROZNAMCHA View */
                    <div className="space-y-6 px-4 lg:px-0 animate-fade-in">
                        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                            <SummaryCard label={isUrdu ? getUrdu('Opening Balance') : "Opening Balance"} value={formatCurrency((reportTotals as any)?.opening)} icon={HandCoins} color="text-slate-600" urdu={isUrdu} />
                            <SummaryCard label={isUrdu ? getUrdu('Inflow') : "Period Inflow"} value={formatCurrency((reportTotals as any)?.inflow)} icon={TrendingUp} color="text-emerald-600" urdu={isUrdu} />
                            <SummaryCard label={isUrdu ? getUrdu('Outflow') : "Period Outflow"} value={formatCurrency((reportTotals as any)?.outflow)} icon={TrendingDown} color="text-rose-600" urdu={isUrdu} />
                            <SummaryCard label={isUrdu ? getUrdu('Closing Balance') : "Net Variance"} value={formatCurrency((reportTotals as any)?.closing)} icon={Activity} color="text-indigo-600" urdu={isUrdu} />
                        </div>
                        <div className="bg-white rounded-3xl border border-slate-200 overflow-hidden shadow-sm">
                            <table className={`w-full text-left border-collapse ${isUrdu ? 'text-right' : ''}`}>
                                <thead className="bg-slate-900 border-b border-white/10 uppercase italic">
                                <tr className={`text-[10px] font-bold text-slate-400 ${isUrdu ? 'urdu-font' : ''}`}>
                                    <th className={`px-6 py-5 ${isUrdu ? 'text-right' : 'text-left'}`}>Date</th>
                                    <th className={`px-6 py-5 ${isUrdu ? 'text-right' : 'text-left'}`}>Activity Narration</th>
                                    <th className="px-6 py-5 text-center">Category</th>
                                    <th className="px-6 py-5 text-center">Reference</th>
                                    <th className={`px-6 py-5 ${isUrdu ? 'text-left' : 'text-right'}`}>Volume</th>
                                </tr>
                                </thead>
                                <tbody className="divide-y divide-slate-100">
                                {currentReportData.map((item: any, idx: number) => (
                                    <tr key={idx} className="hover:bg-slate-50/50 transition-colors">
                                        <td className={`px-6 py-5 text-[10px] font-bold text-slate-500 ${isUrdu ? 'text-right' : 'text-left'}`}>
                                            {item.date}
                                        </td>
                                        <td className={`px-6 py-5 ${isUrdu ? 'urdu-font text-right' : ''}`}>
                                            <p className={`font-bold text-slate-900 text-xs ${isUrdu ? 'urdu-font' : 'uppercase'}`}>{item.desc}</p>
                                            <p className={`text-[9px] font-bold text-slate-400 mt-1 ${isUrdu ? 'urdu-font' : 'uppercase'}`}>{isUrdu ? getUrdu(item.category) : item.category}</p>
                                        </td>
                                        <td className="px-6 py-5 text-center">
                                                <span className={`px-3 py-1 rounded-full text-[8px] font-black uppercase border ${isUrdu ? 'urdu-font' : ''} ${
                                                    item.type === 'Inflow' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' :
                                                        item.type === 'Stock In' ? 'bg-blue-50 text-blue-600 border-blue-100' :
                                                            'bg-rose-50 text-rose-600 border-rose-100'
                                                }`}>{isUrdu ? getUrdu(item.type) : item.type}</span>
                                        </td>
                                        <td className="px-6 py-5 text-center font-mono font-bold text-slate-400 text-[10px]">{item.ref}</td>
                                        <td className={`px-6 py-5 font-black font-mono text-sm ${isUrdu ? 'text-left' : 'text-right'} ${
                                            item.type === 'Inflow' ? 'text-emerald-600' :
                                                item.type === 'Stock In' ? 'text-blue-600' :
                                                    'text-rose-600'
                                        }`}>
                                            {item.type === 'Inflow' ? '+' : item.type === 'Stock In' ? '' : '-'}{formatCurrency(item.amount)}
                                        </td>
                                    </tr>
                                ))}
                                {currentReportData.length === 0 && (
                                    <tr><td colSpan={5} className="p-12 text-center text-slate-400 text-xs italic">No activity recorded for this period</td></tr>
                                )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                ) : !activeReport ? (
                    /* Report Cards Grid */
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 px-4 lg:px-0">
                        {reportCategories.map((cat) => (
                            <div key={cat.id} onClick={() => setActiveReport(cat.id as ReportType)} className={`${cat.id === 'bike_trace' ? 'bg-slate-900 border-slate-800' : 'bg-white border-slate-200'} rounded-[2rem] border p-8 shadow-sm hover:shadow-xl transition-all group flex flex-col items-start cursor-pointer h-full relative overflow-hidden`}>
                                {cat.id === 'bike_trace' && (
                                    <div className="absolute -right-4 -bottom-4 opacity-5 text-white"><Fingerprint size={120}/></div>
                                )}
                                <div className={`w-14 h-14 rounded-2xl ${cat.color} flex items-center justify-center shrink-0 shadow-sm mb-6 transition-transform group-hover:scale-110`}>
                                    <cat.icon size={28} />
                                </div>
                                <h4 className={`text-lg font-black uppercase mb-2 group-hover:text-cyan-600 transition-colors leading-none ${cat.id === 'bike_trace' ? 'text-white' : 'text-slate-900'} ${cat.id === 'urdu_roznamcha' ? 'urdu-font' : ''}`}>{cat.title}</h4>
                                <p className={`text-[11px] font-bold uppercase tracking-tight mb-6 leading-relaxed flex-1 ${cat.id === 'bike_trace' ? 'text-slate-500' : 'text-slate-400'} ${cat.id === 'urdu_roznamcha' ? 'urdu-font' : ''}`}>{cat.desc}</p>
                                <button className={`w-full py-3 text-[9px] font-black uppercase tracking-widest rounded-xl transition-all leading-none ${cat.id === 'bike_trace' ? 'bg-white/5 text-slate-300 group-hover:bg-cyan-600 group-hover:text-white' : 'bg-slate-50 text-slate-500 group-hover:bg-slate-900 group-hover:text-white'}`}>{isUrdu ? 'رپورٹ کھولیں' : 'Execute Search'}</button>
                            </div>
                        ))}
                    </div>
                ) : (
                    /* Other Reports */
                    <div className="space-y-6 px-4 lg:px-0 animate-fade-in">

                        {/* UPDATED CUSTOMER LEDGER VIEW */}
                        {activeReport === 'customer_ledger' && (() => {
                            // Calculate Stats Dynamically for Current View
                            const totalDebits = currentReportData.reduce((acc: number, item: any) => acc + (item.type === 'Invoice' || item.type === 'Booking' ? item.amount : 0), 0);
                            const totalCredits = currentReportData.reduce((acc: number, item: any) => acc + (item.type === 'Payment' ? item.amount : 0), 0);
                            const netBalance = totalDebits - totalCredits;
                            const customer = customers.find((c:any) => String(c.id) === String(selectedCustomerId));

                            return (
                                <div className="space-y-6 animate-fade-in">
                                    {/* Header Cards Grid */}
                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">

                                        {/* 1. Customer Identity Card */}
                                        <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl relative overflow-hidden flex flex-col justify-center">
                                            <div className="absolute right-0 top-0 p-4 opacity-10"><User size={80}/></div>
                                            <div className="flex items-center gap-4 relative z-10">
                                                <div className="w-14 h-14 bg-white/10 rounded-2xl flex items-center justify-center text-cyan-400 border border-white/5 backdrop-blur-sm">
                                                    <User size={28}/>
                                                </div>
                                                <div>
                                                    <p className="text-[10px] font-bold text-cyan-300 uppercase tracking-widest mb-1">Client Profile</p>
                                                    <h3 className="text-xl font-black uppercase leading-none truncate max-w-[180px]">{customer?.name || 'Select Client'}</h3>
                                                    <p className="text-[10px] text-slate-400 mt-1.5 font-bold uppercase">{customer?.contact || 'No Contact Info'}</p>
                                                </div>
                                            </div>
                                        </div>

                                        {/* 2. Lifetime Volume Card */}
                                        <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex flex-col justify-center relative overflow-hidden">
                                            <div className="flex justify-between items-start mb-2">
                                                <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest">Lifetime Invoiced</p>
                                                <div className="p-2 bg-indigo-50 text-indigo-600 rounded-lg">
                                                    <TrendingUp size={16}/>
                                                </div>
                                            </div>
                                            <div className="flex items-end gap-3">
                                                <h3 className="text-3xl font-black text-slate-900 font-mono leading-none tracking-tight">{formatCurrency(totalDebits)}</h3>
                                            </div>
                                            <div className="mt-2 flex items-center gap-2">
                                                <div className="h-1.5 flex-1 bg-slate-100 rounded-full overflow-hidden">
                                                    <div className="h-full bg-emerald-500" style={{width: `${Math.min(100, (totalCredits/totalDebits)*100)}%`}}></div>
                                                </div>
                                                <span className="text-[9px] font-bold text-emerald-600 uppercase">{Math.round((totalCredits/totalDebits)*100) || 0}% Paid</span>
                                            </div>
                                        </div>

                                        {/* 3. Net Balance Card */}
                                        <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex flex-col justify-center">
                                            <div className="flex justify-between items-start mb-2">
                                                <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest">Net Outstanding</p>
                                                <div className={`p-2 rounded-lg ${netBalance > 0 ? 'bg-rose-50 text-rose-600' : 'bg-emerald-50 text-emerald-600'}`}>
                                                    <Wallet size={16}/>
                                                </div>
                                            </div>
                                            <h3 className={`text-4xl font-black font-mono leading-none tracking-tighter ${netBalance > 0 ? 'text-rose-600' : 'text-emerald-600'}`}>
                                                {formatCurrency(netBalance)}
                                            </h3>
                                            <p className="text-[9px] font-bold text-slate-400 uppercase mt-2">
                                                {netBalance > 0 ? 'Payment Required' : 'Account Settled'}
                                            </p>
                                        </div>
                                    </div>

                                    {/* Detailed Ledger Table */}
                                    <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm">
                                        <div className="p-6 bg-slate-50/50 border-b border-slate-100 flex justify-between items-center">
                                            <div className="flex items-center gap-2">
                                                <History size={16} className="text-slate-400"/>
                                                <h4 className="text-xs font-black uppercase tracking-widest text-slate-700">Financial History Log</h4>
                                            </div>
                                            <div className="text-[9px] font-bold text-slate-400 uppercase bg-white px-3 py-1 rounded-full border border-slate-200">
                                                {currentReportData.length} Records Found
                                            </div>
                                        </div>
                                        <table className="w-full text-left">
                                            <thead className="bg-white border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                                            <tr>
                                                <th className="px-8 py-5">Date & Reference</th>
                                                <th className="px-8 py-5">Description</th>
                                                <th className="px-8 py-5 text-right">Debit (Bill)</th>
                                                <th className="px-8 py-5 text-right">Credit (Pay)</th>
                                                <th className="px-8 py-5 text-right">Running Bal</th>
                                            </tr>
                                            </thead>
                                            <tbody className="divide-y divide-slate-100">
                                            {(() => {
                                                let running = 0;
                                                return currentReportData.map((item: any, idx: number) => {
                                                    const isDebit = item.type === 'Invoice' || item.type === 'Booking';
                                                    const amount = item.amount || 0;
                                                    if(isDebit) running += amount; else running -= amount;

                                                    return (
                                                        <tr key={idx} className="hover:bg-slate-50/50 transition-colors group">
                                                            <td className="px-8 py-4">
                                                                <div className="flex flex-col">
                                                                    <span className="text-xs font-bold text-slate-900">{item.date}</span>
                                                                    <span className="text-[9px] font-mono font-bold text-slate-400 mt-0.5 uppercase tracking-wider">{item.ref}</span>
                                                                </div>
                                                            </td>
                                                            <td className="px-8 py-4">
                                                                    <span className={`px-2 py-0.5 rounded text-[7px] font-black uppercase tracking-wider border mb-1 inline-block ${isDebit ? 'bg-indigo-50 text-indigo-600 border-indigo-100' : 'bg-emerald-50 text-emerald-600 border-emerald-100'}`}>
                                                                        {item.type}
                                                                    </span>
                                                                <p className="text-[10px] font-bold text-slate-600 uppercase leading-tight">{item.bike || item.desc || 'General Entry'}</p>
                                                            </td>
                                                            <td className="px-8 py-4 text-right">
                                                                {isDebit ? <span className="text-xs font-black font-mono text-slate-900">{formatCurrency(amount)}</span> : <span className="text-slate-200 font-mono text-[10px]">-</span>}
                                                            </td>
                                                            <td className="px-8 py-4 text-right">
                                                                {!isDebit ? <span className="text-xs font-black font-mono text-emerald-600">{formatCurrency(amount)}</span> : <span className="text-slate-200 font-mono text-[10px]">-</span>}
                                                            </td>
                                                            <td className="px-8 py-4 text-right">
                                                                    <span className={`text-xs font-black font-mono px-3 py-1 rounded-lg ${running > 0 ? 'bg-rose-50 text-rose-600' : 'bg-slate-100 text-slate-500'}`}>
                                                                        {formatCurrency(running)}
                                                                    </span>
                                                            </td>
                                                        </tr>
                                                    );
                                                });
                                            })()}
                                            {currentReportData.length === 0 && (
                                                <tr><td colSpan={5} className="p-12 text-center text-slate-400 text-xs italic font-bold uppercase">Select a customer to view ledger history</td></tr>
                                            )}
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            );
                        })()}

                        {/* FINANCIALS VIEW */}
                        {activeReport === 'financials' && (() => {
                            const revenue = currentReportData.find((d:any) => d.type === 'Revenue')?.value || 0;
                            const cogs = currentReportData.find((d:any) => d.type === 'COGS')?.value || 0;
                            const gross = currentReportData.find((d:any) => d.type === 'Gross')?.value || 0;
                            const opex = currentReportData.find((d:any) => d.type === 'Expense')?.value || 0;
                            const income = currentReportData.find((d:any) => d.type === 'Income')?.value || 0;
                            const net = currentReportData.find((d:any) => d.type === 'Net')?.value || 0;
                            const margin = currentReportData.find((d:any) => d.type === 'Net')?.margin || 0;
                            const topExp = currentReportData.find((d:any) => d.type === 'Expense')?.sub || '';

                            return (
                                <div className="space-y-8 animate-fade-in">
                                    {/* Hero Cards */}
                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                        <div className="bg-white rounded-[2rem] p-8 shadow-sm border border-slate-200">
                                            <div className="flex justify-between items-start mb-4">
                                                <div className="w-12 h-12 bg-indigo-50 text-indigo-600 rounded-2xl flex items-center justify-center"><Activity size={24}/></div>
                                                <span className="text-[10px] font-black uppercase text-indigo-400 tracking-widest bg-indigo-50 px-3 py-1 rounded-full">Top Line</span>
                                            </div>
                                            <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Revenue</p>
                                            <h3 className="text-3xl font-black text-slate-900 font-mono tracking-tight">{formatCurrency(revenue)}</h3>
                                        </div>

                                        <div className="bg-white rounded-[2rem] p-8 shadow-sm border border-slate-200 relative overflow-hidden group">
                                            <div className="flex justify-between items-start mb-4 relative z-10">
                                                <div className="w-12 h-12 bg-emerald-50 text-emerald-600 rounded-2xl flex items-center justify-center"><TrendingUp size={24}/></div>
                                                <span className="text-[10px] font-black uppercase text-emerald-600 tracking-widest bg-emerald-50 px-3 py-1 rounded-full">Gross Profit</span>
                                            </div>
                                            <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1 relative z-10">Sales - Direct Cost</p>
                                            <h3 className="text-3xl font-black text-slate-900 font-mono tracking-tight relative z-10">{formatCurrency(gross)}</h3>
                                        </div>

                                        <div className="bg-slate-900 rounded-[2rem] p-8 shadow-xl text-white relative overflow-hidden">
                                            <div className="absolute -right-4 -bottom-4 opacity-10"><PieChart size={120}/></div>
                                            <div className="flex justify-between items-start mb-4 relative z-10">
                                                <div className="w-12 h-12 bg-white/20 text-white rounded-2xl flex items-center justify-center backdrop-blur-sm"><DollarSign size={24}/></div>
                                                <span className="text-[10px] font-black uppercase text-emerald-300 tracking-widest bg-emerald-500/20 px-3 py-1 rounded-full border border-emerald-500/30">{margin.toFixed(1)}% Margin</span>
                                            </div>
                                            <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1 relative z-10">Net Operating Profit</p>
                                            <h3 className="text-4xl font-black text-white font-mono tracking-tighter relative z-10">{formatCurrency(net)}</h3>
                                        </div>
                                    </div>

                                    {/* Waterfall Table */}
                                    <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm">
                                        <div className="bg-slate-50 px-8 py-4 border-b border-slate-200 flex justify-between items-center">
                                            <h4 className="text-xs font-black uppercase tracking-widest text-slate-600">Profit & Loss Statement (P&L)</h4>
                                            <span className="text-[9px] font-bold text-slate-400 uppercase">Audit Period: {dateFrom} to {dateTo}</span>
                                        </div>
                                        <div className="p-8">
                                            <div className="space-y-4">
                                                <div className="flex justify-between items-center p-4 bg-slate-50/50 rounded-2xl border border-slate-100">
                                                    <div className="flex items-center gap-4">
                                                        <div className="p-2 bg-indigo-100 text-indigo-700 rounded-lg"><Plus size={16}/></div>
                                                        <div>
                                                            <p className="text-xs font-black uppercase text-slate-900">Gross Sales Revenue</p>
                                                            <p className="text-[9px] font-bold text-slate-400 uppercase">Primary Income from Units</p>
                                                        </div>
                                                    </div>
                                                    <p className="text-lg font-black font-mono text-slate-900">{formatCurrency(revenue)}</p>
                                                </div>

                                                <div className="flex justify-between items-center p-4 bg-white rounded-2xl border border-slate-100 relative left-4 w-[calc(100%-1rem)]">
                                                    <div className="flex items-center gap-4">
                                                        <div className="p-2 bg-rose-50 text-rose-600 rounded-lg"><Minus size={16}/></div>
                                                        <div>
                                                            <p className="text-xs font-black uppercase text-slate-700">Cost of Goods Sold (COGS)</p>
                                                            <p className="text-[9px] font-bold text-slate-400 uppercase">Direct Inventory Cost</p>
                                                        </div>
                                                    </div>
                                                    <p className="text-lg font-black font-mono text-rose-600">- {formatCurrency(cogs)}</p>
                                                </div>

                                                <div className="flex justify-between items-center py-3 border-y-2 border-slate-100 ml-8">
                                                    <p className="text-[10px] font-black uppercase text-slate-400 tracking-widest">Gross Profit Subtotal</p>
                                                    <p className="text-xl font-black font-mono text-slate-900">{formatCurrency(gross)}</p>
                                                </div>

                                                <div className="flex justify-between items-center p-4 bg-white rounded-2xl border border-slate-100 relative left-8 w-[calc(100%-2rem)]">
                                                    <div className="flex items-center gap-4">
                                                        <div className="p-2 bg-rose-50 text-rose-600 rounded-lg"><Minus size={16}/></div>
                                                        <div>
                                                            <p className="text-xs font-black uppercase text-slate-700">Operating Expenses</p>
                                                            <p className="text-[9px] font-bold text-slate-400 uppercase">{topExp}</p>
                                                        </div>
                                                    </div>
                                                    <p className="text-lg font-black font-mono text-rose-600">- {formatCurrency(opex)}</p>
                                                </div>

                                                <div className="flex justify-between items-center p-4 bg-white rounded-2xl border border-slate-100 relative left-8 w-[calc(100%-2rem)]">
                                                    <div className="flex items-center gap-4">
                                                        <div className="p-2 bg-emerald-50 text-emerald-600 rounded-lg"><Plus size={16}/></div>
                                                        <div>
                                                            <p className="text-xs font-black uppercase text-slate-700">Ancillary Income</p>
                                                            <p className="text-[9px] font-bold text-slate-400 uppercase">Services, Fees & Claims</p>
                                                        </div>
                                                    </div>
                                                    <p className="text-lg font-black font-mono text-emerald-600">+ {formatCurrency(income)}</p>
                                                </div>

                                                <div className="flex justify-between items-center p-6 bg-slate-900 text-white rounded-2xl mt-4 shadow-xl">
                                                    <div>
                                                        <p className="text-sm font-black uppercase tracking-widest">Net Realized Profit</p>
                                                        <p className="text-[9px] font-bold text-slate-400 uppercase mt-1">EBITDA (Earnings Before Interest/Tax)</p>
                                                    </div>
                                                    <p className="text-3xl font-black font-mono">{formatCurrency(net)}</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            );
                        })()}

                        {/* GENERIC LIST */}
                        {activeReport !== 'financials' && activeReport !== 'stock' && activeReport !== 'sale_invoice' && activeReport !== 'customer_ledger' && activeReport !== 'supplier_ledger' && activeReport !== 'expenses' && activeReport !== 'sales' && activeReport !== 'daily_roznamcha' && activeReport !== 'urdu_roznamcha' && activeReport !== 'bike_trace' && (
                            <div className="bg-white rounded-3xl border border-slate-200 overflow-hidden shadow-sm">
                                <table className="w-full text-left">
                                    <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500"><tr><th className="px-6 py-5">Date & Ref</th><th className="px-6 py-5">Details</th><th className="px-6 py-5 text-right">Amount / Value</th></tr></thead>
                                    <tbody className="divide-y divide-slate-100">{currentReportData.map((item: any, i: number) => (<tr key={i} className="hover:bg-slate-50/50 transition-colors"><td className="px-6 py-5"><p className="font-bold text-slate-900 text-xs">{item.date || item.entry_date || '---'}</p><p className="text-[9px] text-slate-400 font-bold uppercase mt-1">{item.refNumber || item.ref || item.id || '---'}</p></td><td className="px-6 py-5"><p className="font-black text-slate-700 text-xs uppercase">{item.description || item.desc || (typeof item.customer === 'object' ? item.customer.name : item.customer) || item.customerName || item.product || '---'}</p><p className="text-[9px] text-slate-400 font-bold uppercase mt-1">{item.category || item.type || item.notes || ''}</p></td><td className="px-6 py-5 text-right font-black font-mono text-sm text-slate-900">{formatCurrency(item.amount || item.total || item.purchasePrice || item.value || 0)}</td></tr>))}</tbody>
                                </table>
                            </div>
                        )}
                    </div>
                )}
            </div>

            {/* Print Modal Implementation */}
            {isPrintModalOpen && activeReport && !viewingInvoice && (
                <div className="fixed inset-0 bg-slate-900/95 z-[200] overflow-y-auto backdrop-blur-xl animate-fade-in printable-report">
                    <div className="bg-white min-h-screen w-full max-w-5xl mx-auto my-0 lg:my-10 lg:rounded-[3rem] shadow-2xl relative overflow-hidden flex flex-col no-print-margin">
                        <div className="bg-slate-950 p-6 flex justify-between items-center text-white no-print sticky top-0 z-[210]">
                            <div className="flex items-center gap-4">
                                <ChartIcon size={20} className="text-cyan-400" />
                                <h3 className="font-black uppercase tracking-widest text-xs italic">A4 Financial Terminal</h3>
                            </div>
                            <div className="flex gap-3">
                                <button onClick={() => window.print()} className="px-6 py-2 bg-cyan-600 hover:bg-cyan-500 rounded-xl flex items-center gap-2 text-[10px] font-black uppercase tracking-widest transition-all shadow-lg">
                                    <Printer size={14} /> Execute Physical Print
                                </button>
                                <button onClick={() => setIsPrintModalOpen(false)} className="p-2 bg-white/5 hover:bg-rose-500/20 rounded-full transition-all">
                                    <X size={20}/>
                                </button>
                            </div>
                        </div>
                        <div className={`p-16 flex-1 bg-white print-content space-y-12 ${isUrdu ? 'rtl text-right' : ''}`}>
                            <div className={`flex justify-between items-start border-b-4 border-slate-900 pb-10 ${isUrdu ? 'flex-row-reverse' : ''}`}>
                                <div className={`flex items-center gap-6 ${isUrdu ? 'flex-row-reverse' : ''}`}>
                                    <div className="w-20 h-20 bg-slate-900 text-white rounded-3xl flex items-center justify-center">
                                        <Bike size={48} />
                                    </div>
                                    <div>
                                        <h1 className="text-4xl font-black font-logo text-slate-950 uppercase italic leading-none">{showroomSettings.showroomName}</h1>
                                        <p className={`text-[12px] font-black text-slate-400 uppercase tracking-[0.4em] mt-3 ${isUrdu ? 'urdu-font' : ''}`}>{isUrdu ? 'ایگزیکٹو آپریشنل اسٹیٹمنٹ' : 'Executive Operational Statement'}</p>
                                    </div>
                                </div>
                                <div className={isUrdu ? 'text-left' : 'text-right'}>
                                    <h2 className={`text-xl font-black text-slate-900 uppercase tracking-tighter ${isUrdu ? 'urdu-font' : ''}`}>{isUrdu ? getUrdu('Daily Roznamcha') : reportCategories.find(c => c.id === activeReport)?.title}</h2>
                                    <p className={`text-[10px] font-bold text-slate-400 uppercase mt-1 ${isUrdu ? 'urdu-font' : ''}`}>{isUrdu ? getUrdu('Date') : 'Audit Period'}: {dateFrom} to {dateTo}</p>
                                </div>
                            </div>

                            {( (activeReport as string) === 'daily_roznamcha' || (activeReport as string) === 'urdu_roznamcha' ) && (
                                <div className="grid grid-cols-3 gap-8">
                                    <PrintStat label={isUrdu ? getUrdu('Opening Balance') : "OPENING CASH BALANCE"} value={formatCurrency((reportTotals as any)?.opening)} isUrdu={isUrdu} />
                                    <PrintStat label={isUrdu ? getUrdu('Closing Balance') : "CLOSING CASH BALANCE"} value={formatCurrency((reportTotals as any)?.closing)} isHigh isUrdu={isUrdu} />
                                    <PrintStat label={isUrdu ? "تبدیلی" : "NET DAY VARIANCE"} value={formatCurrency(((reportTotals as any)?.inflow || 0) - ((reportTotals as any)?.outflow || 0))} isUrdu={isUrdu} />
                                </div>
                            )}

                            <div className="space-y-4">
                                {activeReport === 'financials' ? (
                                    <div className="space-y-6">
                                        <h3 className="font-black uppercase text-sm border-b border-slate-900 pb-2">Profit & Loss Summary</h3>
                                        <div className="space-y-3">
                                            {currentReportData.map((row: any, i: number) => (
                                                <div key={i} className={`flex justify-between items-center pb-2 border-b border-slate-100 ${row.type === 'Net' ? 'pt-4 border-t-2 border-slate-900 font-black text-lg' : 'text-xs'}`}>
                                                    <span className="uppercase">{row.label}</span>
                                                    <span className="font-mono">{formatCurrency(row.value)}</span>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                ) : (
                                    <table className="w-full text-left border-collapse">
                                        <thead>
                                        <tr className="bg-slate-50 border-y border-slate-200">
                                            <th className={`px-4 py-4 text-[9px] font-black uppercase text-slate-600 ${isUrdu ? 'text-right' : 'text-left'}`}>Date</th>
                                            <th className={`px-4 py-4 text-[9px] font-black uppercase text-slate-600 ${isUrdu ? 'text-right' : 'text-left'}`}>Description</th>
                                            <th className="px-4 py-4 text-[9px] font-black uppercase text-slate-600 text-center">Category</th>
                                            <th className={`px-4 py-4 text-[9px] font-black uppercase text-slate-600 ${isUrdu ? 'text-left' : 'text-right'}`}>Amount</th>
                                        </tr>
                                        </thead>
                                        <tbody className="divide-y divide-slate-100">
                                        {currentReportData.map((item: any, i: number) => (
                                            <tr key={i} className="text-[10px]">
                                                <td className={`px-4 py-4 font-bold text-slate-500 uppercase ${isUrdu ? 'text-right' : 'text-left'}`}>{item.date}</td>
                                                <td className={`px-4 py-4 font-bold uppercase ${isUrdu ? 'text-right' : 'text-left'}`}>{item.desc}</td>
                                                <td className="px-4 py-4 text-center uppercase text-slate-400 font-black tracking-widest">{isUrdu ? getUrdu(item.category) : item.category}</td>
                                                <td className={`px-4 py-4 font-black font-mono ${item.type === 'Inflow' ? 'text-emerald-600' : item.type === 'Stock In' ? 'text-blue-600' : 'text-rose-600'} ${isUrdu ? 'text-left' : 'text-right'}`}>
                                                    {item.type === 'Inflow' ? '+' : item.type === 'Stock In' ? '' : '-'}{formatCurrency(item.amount)}
                                                </td>
                                            </tr>
                                        ))}
                                        </tbody>
                                    </table>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            )}

            {/* --- FULL INVOICE PREVIEW MODAL --- */}
            {viewingInvoice && (
                <div className="fixed inset-0 bg-slate-950/90 z-[250] flex items-center justify-center p-4 backdrop-blur-2xl overflow-y-auto leading-none printable-report">
                    <div className="bg-white rounded-[3rem] w-full max-w-4xl shadow-2xl my-auto animate-fade-in overflow-hidden relative border border-white/20 print-container no-print-margin">
                        <div className="absolute top-0 right-0 p-8 no-print z-50">
                            <button onClick={() => setViewingInvoice(null)} className="p-3 bg-slate-900 text-white rounded-full hover:bg-rose-600 transition-colors shadow-xl"><X size={24}/></button>
                        </div>

                        <div className="p-16 space-y-12 bg-white text-slate-900 print-content">
                            {/* Header Branding */}
                            <div className="flex justify-between items-start border-b-4 border-slate-950 pb-10">
                                <div className="flex items-center gap-6">
                                    <div className="w-20 h-20 bg-slate-950 text-white rounded-[2rem] flex items-center justify-center shadow-2xl print:shadow-none print:border print:border-slate-900">
                                        <Bike size={48} strokeWidth={2.5} />
                                    </div>
                                    <div>
                                        <h1 className="text-5xl font-black font-logo text-slate-950 uppercase italic tracking-tighter leading-[0.85] -ml-1">{showroomSettings.showroomName}</h1>
                                        <p className="text-[10px] font-bold text-slate-400 uppercase tracking-[0.4em] ml-0.5 mt-2">Authorized Sales Invoice</p>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <h2 className="text-2xl font-black text-slate-950 uppercase tracking-tighter italic">Voucher Detail</h2>
                                    <p className="text-[10px] font-black text-cyan-600 uppercase mt-1 font-mono">{viewingInvoice.refNumber || viewingInvoice.id}</p>
                                    <div className={`mt-4 px-4 py-1.5 rounded-full text-[9px] font-black uppercase tracking-[0.2em] border inline-block ${
                                        viewingInvoice.paymentMode === 'Full Payment' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' : 'bg-indigo-50 text-indigo-600 border-indigo-100'
                                    }`}>
                                        {viewingInvoice.paymentMode}
                                    </div>
                                </div>
                            </div>

                            {/* Parties Section */}
                            <div className="grid grid-cols-2 gap-12">
                                <div className="space-y-6">
                                    <div>
                                        <p className="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] mb-3 flex items-center gap-2">Bill To Customer</p>
                                        <h4 className="text-xl font-black text-slate-950 uppercase">{viewingInvoice.customerName}</h4>
                                        <div className="mt-3 space-y-1">
                                            <p className="text-[10px] font-bold text-slate-500 uppercase flex items-center gap-2"><Smartphone size={10}/> {viewingInvoice.customerPhone}</p>
                                        </div>
                                    </div>
                                </div>
                                <div className="text-right space-y-6">
                                    <div>
                                        <p className="text-[9px] font-black text-slate-400 uppercase tracking-[0.2em] mb-3">Transaction Info</p>
                                        <p className="text-xs font-black text-slate-900 uppercase">Date of Issue: <span className="text-slate-500 font-mono">{viewingInvoice.date}</span></p>
                                        <p className="text-xs font-black text-slate-900 uppercase mt-2">Ref: <span className="text-slate-500 font-mono">{viewingInvoice.refNumber || 'N/A'}</span></p>
                                    </div>
                                </div>
                            </div>

                            {/* Product Specs Grid */}
                            <div className="bg-slate-50 rounded-[2.5rem] p-10 border border-slate-100 relative overflow-hidden">
                                <div className="absolute top-0 left-0 w-1 h-full bg-slate-950"></div>
                                <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
                                    <div className="col-span-2">
                                        <p className="text-[8px] font-black text-slate-400 uppercase tracking-widest mb-1.5">Description of Goods</p>
                                        <p className="text-lg font-black text-slate-900 uppercase">{viewingInvoice.productName}</p>
                                    </div>
                                    <div>
                                        <p className="text-[8px] font-black text-slate-400 uppercase tracking-widest mb-1.5">Engine Serial</p>
                                        <p className="text-xs font-black font-mono text-slate-900 uppercase tracking-wider">{viewingInvoice.engine || '---'}</p>
                                    </div>
                                    <div>
                                        <p className="text-[8px] font-black text-slate-400 uppercase tracking-widest mb-1.5">Chassis Serial</p>
                                        <p className="text-xs font-black font-mono text-slate-900 uppercase tracking-wider">{viewingInvoice.chassis || '---'}</p>
                                    </div>
                                </div>
                            </div>

                            {/* Financial Summary Table */}
                            <div className="space-y-4">
                                <table className="w-full text-left">
                                    <thead className="bg-slate-950 text-white rounded-xl">
                                    <tr className="text-[9px] font-black uppercase tracking-[0.2em]">
                                        <th className="px-6 py-4">Financial Ledger Item</th>
                                        <th className="px-6 py-4 text-right">Debit Volume (PKR)</th>
                                    </tr>
                                    </thead>
                                    <tbody className="divide-y divide-slate-100">
                                    <tr>
                                        <td className="px-6 py-6 text-xs font-bold text-slate-600 uppercase">Net Unit Agreement Total</td>
                                        <td className="px-6 py-6 text-right font-black font-mono text-slate-950 text-lg">{formatCurrency(viewingInvoice.total)}</td>
                                    </tr>
                                    <tr>
                                        <td className="px-6 py-6 text-xs font-bold text-emerald-600 uppercase">Total Received (Advance/Cash)</td>
                                        <td className="px-6 py-6 text-right font-black font-mono text-emerald-600 text-lg">{formatCurrency(viewingInvoice.paid)}</td>
                                    </tr>
                                    </tbody>
                                    <tfoot className="border-t-4 border-slate-950">
                                    <tr className="bg-slate-50">
                                        <td className="px-6 py-6 text-xs font-black text-slate-950 uppercase tracking-widest">Net Balance Due</td>
                                        <td className="px-6 py-6 text-right font-black font-mono text-2xl text-slate-950">{formatCurrency(viewingInvoice.balance)}</td>
                                    </tr>
                                    </tfoot>
                                </table>
                            </div>

                            {/* Footer / Auth */}
                            <div className="pt-16 flex justify-between items-end">
                                <div className="max-w-xs space-y-4">
                                    <div className="flex items-center gap-3">
                                        <ShieldAlert className="text-emerald-500" size={16}/>
                                        <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest">Data Verification Hash: {btoa(viewingInvoice.id).slice(0, 12)}</p>
                                    </div>
                                    <p className="text-[8px] text-slate-400 font-bold uppercase leading-relaxed italic">The above unit is sold in good working condition. Registration responsibility lies with the dealership agent unless specified.</p>
                                </div>
                                <div className="text-center w-48 space-y-3">
                                    <div className="h-[1px] bg-slate-300 w-full mb-3"></div>
                                    <p className="text-[10px] font-black text-slate-950 uppercase tracking-widest">Authorized Signature</p>
                                </div>
                            </div>

                            <div className="pt-12 text-center border-t border-dashed border-slate-200 no-print">
                                <button onClick={() => window.print()} className="px-8 py-4 bg-slate-950 text-white rounded-2xl text-[10px] font-black uppercase tracking-[0.2em] shadow-xl hover:bg-black transition-all flex items-center justify-center gap-3 mx-auto">
                                    <Printer size={16}/> Execute Physical Print
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            <style>{`
                .rtl { direction: rtl; }
                .urdu-font { font-family: 'Noto Nastaliq Urdu', serif; }
                @media print {
                    .no-print { display: none !important; }
                    .print-content { padding: 0 !important; margin: 0 !important; }
                    .printable-report {
                        position: absolute !important;
                        top: 0 !important;
                        left: 0 !important;
                        width: 100% !important;
                        height: auto !important;
                        background: white !important;
                        z-index: 9999 !important;
                        overflow: visible !important;
                    }
                    body { background: white !important; margin: 0; }
                    .shadow-2xl { box-shadow: none !important; }
                    .no-print-margin { margin: 0 !important; border-radius: 0 !important; }
                    .rtl { direction: rtl !important; }
                    @page {
                        size: A4 portrait;
                        margin: 0;
                    }
                }
            `}</style>
        </div>
    );
}

function SummaryCard({ label, value, icon: Icon, color, urdu }: any) {
    return (
        <div className={`bg-white p-6 rounded-2xl border border-slate-200 shadow-sm flex items-center gap-4 transition-all hover:shadow-md ${urdu ? 'flex-row-reverse text-right' : ''}`}>
            <div className={`w-12 h-12 rounded-xl bg-slate-50 ${color} flex items-center justify-center shrink-0 shadow-inner`}>
                <Icon size={24} />
            </div>
            <div className="leading-none flex-1">
                <p className={`text-[10px] font-black uppercase text-slate-400 tracking-widest mb-1.5 ${urdu ? 'urdu-font' : ''}`}>{label}</p>
                <p className={`text-lg font-black font-mono tracking-tight text-slate-900`}>{value}</p>
            </div>
        </div>
    );
}

function PrintStat({ label, value, isHigh, isUrdu }: any) {
    return (
        <div className="bg-slate-50 p-6 rounded-3xl border border-slate-100 shadow-inner text-center">
            <p className={`text-[9px] font-black text-slate-400 uppercase tracking-widest mb-2 leading-none ${isUrdu ? 'urdu-font' : ''}`}>{label}</p>
            <p className={`text-2xl font-black font-mono leading-none ${isHigh ? 'text-emerald-600 underline' : 'text-slate-900'}`}>{value}</p>
        </div>
    );
}
