
import React, { useState, useMemo } from 'react';
import { router } from '@inertiajs/react';
import {
    Search, User, IdCard, Smartphone, MapPin, StickyNote,
    X, Plus, Pencil, Trash2, BadgeCheck, Hash, Users, ShieldCheck,
    TrendingUp, Wallet, ShoppingBag, CalendarClock, FileText, ArrowRightLeft,
    CheckCircle2, AlertTriangle, Phone, Crown
} from 'lucide-react';
import { IntegratedInput, getCurrentDate, formatCurrency } from './CommonUI.tsx';

export function CustomersPage({ customers, salesHistory, bookings, notify }: any) {
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [isLedgerOpen, setIsLedgerOpen] = useState(false);
    const [editingId, setEditingId] = useState<string | null>(null);
    const [viewingCustomer, setViewingCustomer] = useState<any>(null);
    const [searchTerm, setSearchTerm] = useState('');

    // Default Form State
    const [formData, setFormData] = useState({
        name: '', fatherName: '', cnic: '', contact: '', address: '',
        guarantorName: '', guarantorFatherName: '', guarantorAddress: '',
        notes: '', refNumber: ''
    });

    // --- Financial Engine ---
    // Aggregates sales and payments to determine customer worth and debt
    const customerFinancials = useMemo(() => {
        const stats: Record<string, any> = {};

        // Initialize
        customers.forEach((c: any) => {
            stats[c.id] = {
                totalPurchased: 0,
                totalPaid: 0,
                balance: 0,
                lastActivity: 'N/A',
                purchaseCount: 0,
                rating: 'New'
            };
        });

        // Process Sales History
        (salesHistory || []).forEach((sale: any) => {
            const cId = sale.customerId;
            if (stats[cId]) {
                const total = Number(sale.total) || 0;
                const paid = (sale.payments || []).reduce((sum:number, p:any) => sum + (Number(p.amount) || 0), 0);

                stats[cId].totalPurchased += total;
                stats[cId].totalPaid += paid;
                stats[cId].balance += (total - paid);
                stats[cId].purchaseCount += 1;

                // Track latest date
                if (stats[cId].lastActivity === 'N/A' || new Date(sale.date) > new Date(stats[cId].lastActivity)) {
                    stats[cId].lastActivity = sale.date;
                }
            }
        });

        // Determine Rating
        Object.keys(stats).forEach(key => {
            const s = stats[key];
            if (s.totalPurchased > 500000) s.rating = 'Platinum';
            else if (s.totalPurchased > 150000) s.rating = 'Gold';
            else if (s.purchaseCount > 0) s.rating = 'Active';
        });

        return stats;
    }, [customers, salesHistory]);

    // --- KPI Metrics ---
    const kpis = useMemo(() => {
        const totalReceivables = Object.values(customerFinancials).reduce((acc:number, cur:any) => acc + cur.balance, 0);
        const activeCount = Object.values(customerFinancials).filter((c:any) => c.purchaseCount > 0).length;
        const totalVolume = Object.values(customerFinancials).reduce((acc:number, cur:any) => acc + cur.totalPurchased, 0);

        return { totalReceivables, activeCount, totalVolume };
    }, [customerFinancials]);

    // --- Filtering ---
    const filteredCustomers = useMemo(() => {
        const term = searchTerm.toLowerCase();
        return customers.filter((c: any) =>
            c.name.toLowerCase().includes(term) ||
            (c.contact && c.contact.includes(term)) ||
            (c.cnic && c.cnic.includes(term))
        ).sort((a:any, b:any) => {
            // Sort by balance descending (debtors first)
            const balA = customerFinancials[a.id]?.balance || 0;
            const balB = customerFinancials[b.id]?.balance || 0;
            return balB - balA;
        });
    }, [customers, searchTerm, customerFinancials]);

    // --- Ledger Logic ---
    const customerLedger = useMemo(() => {
        if (!viewingCustomer) return [];
        const cid = viewingCustomer.id;
        const sales = (salesHistory || []).filter((s:any) => s.customerId === cid);
        const customerBookings = (bookings || []).filter((b:any) => b.customerId === cid);

        const ledger: any[] = [];

        // 1. Sales (Invoices)
        sales.forEach((s: any) => {
            ledger.push({
                date: s.date,
                type: 'Invoice',
                desc: `Purchase: ${s.bike || 'Vehicle'}`,
                ref: s.refNumber || `INV-${s.id.toString().substring(0,6)}`,
                debit: s.total,
                credit: 0
            });

            // 2. Payments linked to sales
            (s.payments || []).forEach((p: any) => {
                ledger.push({
                    date: p.date,
                    type: 'Payment',
                    desc: `Payment for ${s.bike}`,
                    ref: `RCPT-${p.id}`,
                    debit: 0,
                    credit: p.amount
                });
            });
        });

        // 3. Bookings
        customerBookings.forEach((b: any) => {
            if(b.status !== 'Fulfilled') { // Fulfilled bookings become sales, don't double count
                ledger.push({
                    date: b.date,
                    type: 'Booking',
                    desc: `Reservation: ${b.productName}`,
                    ref: `BKG-${b.id}`,
                    debit: b.total,
                    credit: b.advance
                });
            }
        });

        return ledger.sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());
    }, [viewingCustomer, salesHistory, bookings]);


    // --- Actions ---
    const handleSave = () => {
        if (!formData.name) return;
        const data = editingId ? { ...formData, id: editingId } : formData;
        router.post('/customers', data, {
            onSuccess: () => {
                notify(editingId ? 'Client Profile Updated' : 'New Client Onboarded');
                closeModal();
            }
        });
    };

    const handleDelete = (id: string) => {
        if (confirm(`Archive customer record? This action cannot be undone.`)) {
            router.delete(`/customers/${id}`, {
                onSuccess: () => notify('Customer Removed Successfully')
            });
        }
    };

    const openEdit = (customer: any) => {
        setEditingId(customer.id);
        setFormData({ ...customer, refNumber: customer.refNumber || '' });
        setIsModalOpen(true);
    };

    const openLedger = (customer: any) => {
        setViewingCustomer(customer);
        setIsLedgerOpen(true);
    };

    const closeModal = () => {
        setIsModalOpen(false);
        setEditingId(null);
        setFormData({ name: '', fatherName: '', cnic: '', contact: '', address: '', guarantorName: '', guarantorFatherName: '', guarantorAddress: '', notes: '', refNumber: '' });
    };

    return (
        <div className="animate-fade-in space-y-8 pb-10">
            {/* KPI Dashboard */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl flex items-center gap-4 relative overflow-hidden group">
                    <div className="absolute right-0 top-0 p-4 opacity-10 group-hover:scale-110 transition-transform"><Users size={80}/></div>
                    <div className="w-12 h-12 bg-indigo-500 rounded-2xl flex items-center justify-center text-white shadow-lg"><Wallet size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-indigo-300 uppercase tracking-widest mb-1">Total Receivables</p>
                        <h3 className="text-3xl font-black font-mono tracking-tight">{formatCurrency(kpis.totalReceivables)}</h3>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-emerald-50 text-emerald-600 rounded-xl flex items-center justify-center shadow-inner"><TrendingUp size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Lifetime Volume</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(kpis.totalVolume)}</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-cyan-50 text-cyan-600 rounded-xl flex items-center justify-center shadow-inner"><BadgeCheck size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Active Accounts</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{kpis.activeCount}</p>
                    </div>
                </div>
            </div>

            {/* Controls */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div>
                    <h3 className="text-lg font-bold font-heading text-slate-900 uppercase italic">Client Directory</h3>
                    <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest mt-1">Relationship Management</p>
                </div>
                <div className="flex gap-2 w-full sm:w-auto">
                    <div className="relative flex-1 sm:w-64">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-slate-400" size={14} />
                        <input type="text" placeholder="Search Name, Phone, CNIC..." className="w-full pl-9 pr-4 py-2.5 bg-white border border-slate-200 rounded-xl text-xs font-bold uppercase tracking-widest outline-none shadow-sm focus:border-indigo-500 transition-all" value={searchTerm} onChange={(e) => setSearchTerm(e.target.value)} />
                    </div>
                    <button onClick={() => setIsModalOpen(true)} className="px-6 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-bold uppercase tracking-widest hover:bg-black flex items-center gap-2 shadow-lg whitespace-nowrap">
                        <Plus size={14} /> Onboard Client
                    </button>
                </div>
            </div>

            {/* Enhanced Customer List */}
            <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                <table className="w-full text-left">
                    <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                    <tr>
                        <th className="px-8 py-5">Profile Identity</th>
                        <th className="px-8 py-5">Contact & Location</th>
                        <th className="px-8 py-5">Engagement Level</th>
                        <th className="px-8 py-5 text-right">Outstanding Balance</th>
                        <th className="px-8 py-5 text-right">Actions</th>
                    </tr>
                    </thead>
                    <tbody className="divide-y divide-slate-100">
                    {filteredCustomers.map((c: any) => {
                        const fin = customerFinancials[c.id] || {};
                        const hasDebt = fin.balance > 0;

                        return (
                            <tr key={c.id} className="hover:bg-slate-50/50 transition-colors group">
                                <td className="px-8 py-5">
                                    <div className="flex items-center gap-4">
                                        <div className={`w-10 h-10 rounded-full flex items-center justify-center text-white font-bold text-xs ${
                                            fin.rating === 'Platinum' ? 'bg-indigo-600 shadow-md shadow-indigo-200' :
                                                fin.rating === 'Gold' ? 'bg-amber-500 shadow-md shadow-amber-200' :
                                                    'bg-slate-300'
                                        }`}>
                                            {fin.rating === 'Platinum' || fin.rating === 'Gold' ? <Crown size={16}/> : <User size={16}/>}
                                        </div>
                                        <div>
                                            <div className="flex items-center gap-2">
                                                <p className="font-black text-slate-900 text-sm uppercase">{c.name}</p>
                                                {fin.rating !== 'New' && (
                                                    <span className={`text-[7px] font-black uppercase px-1.5 py-0.5 rounded border ${
                                                        fin.rating === 'Platinum' ? 'bg-indigo-50 text-indigo-600 border-indigo-100' :
                                                            fin.rating === 'Gold' ? 'bg-amber-50 text-amber-600 border-amber-100' :
                                                                'bg-emerald-50 text-emerald-600 border-emerald-100'
                                                    }`}>{fin.rating}</span>
                                                )}
                                            </div>
                                            <p className="text-[10px] text-slate-400 font-bold uppercase mt-0.5 flex items-center gap-1">
                                                <IdCard size={10}/> {c.cnic || 'No CNIC'}
                                            </p>
                                        </div>
                                    </div>
                                </td>
                                <td className="px-8 py-5">
                                    <div className="space-y-1">
                                        <p className="text-[10px] font-bold text-slate-700 font-mono flex items-center gap-1"><Smartphone size={10} className="text-slate-400"/> {c.contact}</p>
                                        <p className="text-[9px] font-bold text-slate-400 uppercase truncate max-w-[150px]" title={c.address}>{c.address || 'No Address'}</p>
                                    </div>
                                </td>
                                <td className="px-8 py-5">
                                    <div>
                                        <p className="text-[10px] font-black text-slate-600 uppercase">{fin.purchaseCount} Transactions</p>
                                        <p className="text-[9px] font-bold text-slate-400 uppercase mt-0.5">Last: {fin.lastActivity}</p>
                                    </div>
                                </td>
                                <td className="px-8 py-5 text-right">
                                    <p className={`font-black font-mono text-sm ${hasDebt ? 'text-rose-600' : 'text-emerald-600'}`}>
                                        {formatCurrency(fin.balance)}
                                    </p>
                                    <p className="text-[8px] font-bold text-slate-400 uppercase tracking-widest mt-0.5">
                                        Lifetime: {formatCurrency(fin.totalPurchased)}
                                    </p>
                                </td>
                                <td className="px-8 py-5 text-right">
                                    <div className="flex justify-end gap-1 opacity-0 group-hover:opacity-100 transition-opacity">
                                        <button onClick={() => openLedger(c)} className="p-2.5 bg-slate-50 text-slate-500 hover:bg-slate-900 hover:text-white rounded-xl transition-all shadow-sm" title="View Ledger">
                                            <FileText size={14}/>
                                        </button>
                                        <button onClick={() => openEdit(c)} className="p-2.5 bg-slate-50 text-slate-400 hover:text-cyan-600 hover:bg-cyan-50 rounded-xl transition-all"><Pencil size={14}/></button>
                                        <button onClick={() => handleDelete(c.id)} className="p-2.5 bg-slate-50 text-slate-400 hover:text-rose-600 hover:bg-rose-50 rounded-xl transition-all"><Trash2 size={14}/></button>
                                    </div>
                                </td>
                            </tr>
                        );
                    })}
                    {filteredCustomers.length === 0 && (
                        <tr><td colSpan={5} className="p-12 text-center text-slate-400 text-xs font-bold uppercase italic">No matching client records found</td></tr>
                    )}
                    </tbody>
                </table>
            </div>

            {/* CREATE/EDIT MODAL */}
            {isModalOpen && (
                <div className="fixed inset-0 bg-slate-900/60 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-3xl p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex items-center justify-between border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-slate-900 rounded-2xl flex items-center justify-center text-white"><User size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest">{editingId ? 'Edit Profile' : 'New Client Registration'}</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1">KYC & Contact Info</p>
                                </div>
                            </div>
                            <button onClick={closeModal} className="p-2 hover:bg-slate-100 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div className="space-y-4">
                                <h4 className="text-[10px] font-black text-slate-400 uppercase tracking-widest border-b border-slate-100 pb-2">Primary Identity</h4>
                                <IntegratedInput label="Full Name" value={formData.name} onChange={(e: any) => setFormData({...formData, name: e.target.value})} icon={User} />
                                <IntegratedInput label="Father / Husband Name" value={formData.fatherName} onChange={(e: any) => setFormData({...formData, fatherName: e.target.value})} icon={User} />
                                <IntegratedInput label="CNIC Number" value={formData.cnic} onChange={(e: any) => setFormData({...formData, cnic: e.target.value})} icon={IdCard} />
                                <IntegratedInput label="Contact Mobile" value={formData.contact} onChange={(e: any) => setFormData({...formData, contact: e.target.value})} icon={Phone} />
                                <IntegratedInput label="Residential Address" value={formData.address} onChange={(e: any) => setFormData({...formData, address: e.target.value})} icon={MapPin} />
                            </div>
                            <div className="space-y-4">
                                <h4 className="text-[10px] font-black text-slate-400 uppercase tracking-widest border-b border-slate-100 pb-2">Guarantor / Next of Kin</h4>
                                <IntegratedInput label="Guarantor Name" value={formData.guarantorName} onChange={(e: any) => setFormData({...formData, guarantorName: e.target.value})} icon={ShieldCheck} />
                                <IntegratedInput label="Guarantor Father Name" value={formData.guarantorFatherName} onChange={(e: any) => setFormData({...formData, guarantorFatherName: e.target.value})} icon={User} />
                                <IntegratedInput label="Guarantor Address" value={formData.guarantorAddress} onChange={(e: any) => setFormData({...formData, guarantorAddress: e.target.value})} icon={MapPin} />
                                <IntegratedInput label="Internal Notes" value={formData.notes} onChange={(e: any) => setFormData({...formData, notes: e.target.value})} icon={StickyNote} placeholder="Credit limit, behavior notes..." />
                            </div>
                        </div>

                        <button onClick={handleSave} className="w-full bg-slate-900 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest hover:bg-black transition-all shadow-xl flex items-center justify-center gap-2">
                            <BadgeCheck size={18} /> {editingId ? 'Update Client Record' : 'Create Client Profile'}
                        </button>
                    </div>
                </div>
            )}

            {/* LEDGER MODAL */}
            {isLedgerOpen && viewingCustomer && (
                <div className="fixed inset-0 bg-slate-950/80 z-[120] flex items-center justify-center p-4 backdrop-blur-md overflow-y-auto leading-none">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-4xl shadow-2xl my-auto animate-fade-in relative flex flex-col max-h-[85vh] overflow-hidden">
                        <div className="bg-slate-950 p-8 shrink-0 flex justify-between items-center text-white">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-white/10 rounded-2xl flex items-center justify-center text-cyan-400"><TrendingUp size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">{viewingCustomer.name}</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1 tracking-widest">Complete Financial History</p>
                                </div>
                            </div>
                            <button onClick={() => { setIsLedgerOpen(false); setViewingCustomer(null); }} className="p-2 hover:bg-white/10 rounded-full text-slate-400 hover:text-white transition-colors"><X size={20} /></button>
                        </div>

                        {/* Ledger Summary Stats */}
                        <div className="grid grid-cols-3 gap-4 p-6 bg-slate-50 border-b border-slate-200 shrink-0">
                            <div className="text-center">
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Invoiced</p>
                                <p className="text-xl font-black font-mono text-slate-900">{formatCurrency(customerFinancials[viewingCustomer.id]?.totalPurchased || 0)}</p>
                            </div>
                            <div className="text-center border-x border-slate-200">
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Paid</p>
                                <p className="text-xl font-black font-mono text-emerald-600">{formatCurrency(customerFinancials[viewingCustomer.id]?.totalPaid || 0)}</p>
                            </div>
                            <div className="text-center">
                                <p className="text-[9px] font-black text-slate-400 uppercase tracking-widest mb-1">Current Due</p>
                                <p className="text-xl font-black font-mono text-rose-600">{formatCurrency(customerFinancials[viewingCustomer.id]?.balance || 0)}</p>
                            </div>
                        </div>

                        <div className="flex-1 overflow-y-auto custom-scrollbar p-0 bg-white">
                            <table className="w-full text-left">
                                <thead className="bg-slate-50 text-[9px] font-black uppercase text-slate-500 sticky top-0 border-b border-slate-100">
                                <tr>
                                    <th className="px-6 py-4">Date & Ref</th>
                                    <th className="px-6 py-4">Description</th>
                                    <th className="px-6 py-4 text-right text-rose-600">Debit (Sale)</th>
                                    <th className="px-6 py-4 text-right text-emerald-600">Credit (Pay)</th>
                                </tr>
                                </thead>
                                <tbody className="divide-y divide-slate-100">
                                {customerLedger.map((txn: any, idx: number) => (
                                    <tr key={idx} className="hover:bg-slate-50 transition-colors">
                                        <td className="px-6 py-4">
                                            <p className="text-xs font-bold text-slate-700">{txn.date}</p>
                                            <p className="text-[9px] font-bold text-slate-400 font-mono mt-0.5 uppercase">{txn.ref}</p>
                                        </td>
                                        <td className="px-6 py-4">
                                            <p className="text-[10px] font-black uppercase text-slate-600">{txn.type}</p>
                                            <p className="text-[10px] text-slate-500">{txn.desc}</p>
                                        </td>
                                        <td className="px-6 py-4 text-right font-mono font-bold text-xs text-rose-600">
                                            {txn.debit > 0 ? formatCurrency(txn.debit) : '-'}
                                        </td>
                                        <td className="px-6 py-4 text-right font-mono font-bold text-xs text-emerald-600">
                                            {txn.credit > 0 ? formatCurrency(txn.credit) : '-'}
                                        </td>
                                    </tr>
                                ))}
                                {customerLedger.length === 0 && (
                                    <tr><td colSpan={4} className="p-12 text-center text-slate-300 text-xs font-bold uppercase italic">No history found for this client</td></tr>
                                )}
                                </tbody>
                            </table>
                        </div>

                        <div className="p-4 bg-slate-50 border-t border-slate-200 text-center text-[9px] font-bold text-slate-400 uppercase tracking-widest shrink-0">
                            End of Financial Record
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
