

import React, { useState, useMemo } from 'react';
import {
    Briefcase, TrendingUp, ArrowUpRight, ArrowDownLeft, Plus,
    Calendar, DollarSign, User, StickyNote, Landmark, CheckCircle2,
    AlertCircle, X, Hash, PieChart, Wallet, Clock
} from 'lucide-react';
import { router } from '@inertiajs/react';
import { IntegratedInput, IntegratedSelect, formatCurrency, getCurrentDate } from './CommonUI.tsx';

export default function BusinessAdvancesPage({ advances, accounts, notify }: any) {
    const [isLendModalOpen, setIsLendModalOpen] = useState(false);
    const [isReturnModalOpen, setIsReturnModalOpen] = useState(false);
    const [selectedAdvance, setSelectedAdvance] = useState<any>(null);

    // Lending Form
    const [lendForm, setLendForm] = useState({
        partner_name: '',
        contact: '',
        amount: '',
        expected_return: '',
        account_id: accounts?.[0]?.id || '',
        issue_date: getCurrentDate(),
        due_date: '',
        ref_number: '',
        notes: ''
    });

    // Return Form
    const [returnForm, setReturnForm] = useState({
        amount: '',
        account_id: accounts?.[0]?.id || '',
        date: getCurrentDate(),
        is_final_settlement: false,
        notes: ''
    });

    // --- Stats ---
    const stats = useMemo(() => {
        const totalPrincipal = advances.reduce((acc: number, a: any) => acc + Number(a.principal_amount), 0);
        const totalReturned = advances.reduce((acc: number, a: any) => acc + Number(a.returned_amount), 0);
        const outstanding = Math.max(0, totalPrincipal - totalReturned); // Rough estimate of exposure
        const profit = advances.reduce((acc: number, a: any) => acc + Number(a.profit_realized), 0);

        return { totalPrincipal, totalReturned, outstanding, profit };
    }, [advances]);

    // --- Handlers ---
    const handleLend = () => {
        if(!lendForm.partner_name || !lendForm.amount) return;
        router.post('/advances', lendForm, {
            onSuccess: () => {
                setIsLendModalOpen(false);
                setLendForm({ partner_name: '', contact: '', amount: '', expected_return: '', account_id: accounts?.[0]?.id || '', issue_date: getCurrentDate(), due_date: '', ref_number: '', notes: '' });
                if(notify) notify("Funds Advanced Successfully");
            }
        });
    };

    const handleReturn = () => {
        if(!selectedAdvance || !returnForm.amount) return;
        router.post(`/advances/${selectedAdvance.id}/return`, returnForm, {
            onSuccess: () => {
                setIsReturnModalOpen(false);
                setSelectedAdvance(null);
                setReturnForm({ amount: '', account_id: accounts?.[0]?.id || '', date: getCurrentDate(), is_final_settlement: false, notes: '' });
                if(notify) notify("Return Processed");
            }
        });
    };

    const openReturnModal = (advance: any) => {
        setSelectedAdvance(advance);
        setIsReturnModalOpen(true);
    };

    return (
        <div className="animate-fade-in space-y-8 pb-20">
            {/* KPI Header */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 px-4 lg:px-0">
                <div className="bg-slate-900 rounded-[2rem] p-6 text-white shadow-xl flex items-center gap-4 relative overflow-hidden">
                    <div className="absolute top-0 right-0 p-4 opacity-10"><Briefcase size={80}/></div>
                    <div className="w-12 h-12 bg-indigo-500 rounded-2xl flex items-center justify-center text-white shadow-lg"><ArrowUpRight size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-indigo-300 uppercase tracking-widest mb-1">Active Exposure</p>
                        <h3 className="text-3xl font-black font-mono tracking-tight">{formatCurrency(stats.outstanding)}</h3>
                        <p className="text-[9px] text-slate-400 uppercase mt-1">Total Principal Deployed</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-emerald-50 text-emerald-600 rounded-xl flex items-center justify-center shadow-inner"><TrendingUp size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Realized Profit</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(stats.profit)}</p>
                        <p className="text-[8px] font-bold text-emerald-500 uppercase">ROI from Advances</p>
                    </div>
                </div>
                <div className="bg-white rounded-[2rem] p-6 border border-slate-200 shadow-sm flex items-center gap-4">
                    <div className="w-12 h-12 bg-blue-50 text-blue-600 rounded-xl flex items-center justify-center shadow-inner"><Wallet size={24}/></div>
                    <div>
                        <p className="text-[10px] font-black text-slate-400 uppercase tracking-widest mb-1">Total Recovered</p>
                        <p className="text-xl font-black text-slate-900 font-mono">{formatCurrency(stats.totalReturned)}</p>
                    </div>
                </div>
            </div>

            {/* Controls */}
            <div className="flex flex-col sm:flex-row justify-between items-center gap-4 px-4 lg:px-0">
                <div>
                    <h3 className="text-lg font-bold font-heading text-slate-900 uppercase italic">Business Advances</h3>
                    <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest mt-1">Short-term B2B Lending & Investment</p>
                </div>
                <button onClick={() => setIsLendModalOpen(true)} className="px-6 py-2.5 bg-slate-900 text-white rounded-xl text-xs font-bold uppercase tracking-widest shadow-lg hover:bg-black transition-all flex items-center gap-2">
                    <Plus size={14} /> New Advance
                </button>
            </div>

            {/* Advances Table */}
            <div className="bg-white rounded-[2.5rem] border border-slate-200 overflow-hidden shadow-sm mx-4 lg:mx-0">
                <table className="w-full text-left">
                    <thead className="bg-slate-50 border-b border-slate-100 uppercase italic text-[10px] font-bold text-slate-500">
                    <tr>
                        <th className="px-8 py-5">Business Partner</th>
                        <th className="px-8 py-5">Timeline</th>
                        <th className="px-8 py-5 text-center">Status</th>
                        <th className="px-8 py-5 text-right">Financial Position</th>
                        <th className="px-8 py-5 text-right">Action</th>
                    </tr>
                    </thead>
                    <tbody className="divide-y divide-slate-100">
                    {advances.map((adv: any) => {
                        const principal = Number(adv.principal_amount);
                        const returned = Number(adv.returned_amount);
                        const profit = Number(adv.profit_realized);
                        const netBalance = principal - returned; // Negative implies profit taken out

                        return (
                            <tr key={adv.id} className="hover:bg-slate-50/50 transition-colors group">
                                <td className="px-8 py-5">
                                    <div className="flex items-center gap-3">
                                        <div className="w-10 h-10 rounded-xl bg-slate-100 flex items-center justify-center text-slate-500"><Briefcase size={18}/></div>
                                        <div>
                                            <p className="font-black text-slate-900 text-sm uppercase">{adv.partner_name}</p>
                                            <p className="text-[10px] text-slate-400 font-bold uppercase mt-0.5">{adv.contact_info || 'No Contact'}</p>
                                        </div>
                                    </div>
                                </td>
                                <td className="px-8 py-5">
                                    <p className="text-[10px] font-bold text-slate-500 uppercase">Issued: {adv.issue_date}</p>
                                    <p className={`text-[10px] font-black uppercase mt-1 ${adv.due_date ? 'text-rose-500' : 'text-slate-300'}`}>
                                        Due: {adv.due_date || 'Open Ended'}
                                    </p>
                                </td>
                                <td className="px-8 py-5 text-center">
                                        <span className={`px-3 py-1 rounded-full text-[8px] font-black uppercase border tracking-widest ${
                                            adv.status === 'Settled' ? 'bg-emerald-50 text-emerald-600 border-emerald-100' : 'bg-indigo-50 text-indigo-600 border-indigo-100'
                                        }`}>
                                            {adv.status}
                                        </span>
                                </td>
                                <td className="px-8 py-5 text-right">
                                    <div className="flex flex-col items-end">
                                        <p className="text-xs font-black font-mono text-slate-900">{formatCurrency(principal)}</p>
                                        <div className="w-full h-px bg-slate-200 my-1"></div>
                                        <div className="flex items-center gap-2">
                                            <span className="text-[8px] font-bold uppercase text-slate-400">Returned</span>
                                            <span className="text-[10px] font-mono font-bold text-emerald-600">{formatCurrency(returned)}</span>
                                        </div>
                                        {profit > 0 && (
                                            <div className="flex items-center gap-2">
                                                <span className="text-[8px] font-bold uppercase text-slate-400">Profit</span>
                                                <span className="text-[10px] font-mono font-bold text-indigo-600">+{formatCurrency(profit)}</span>
                                            </div>
                                        )}
                                    </div>
                                </td>
                                <td className="px-8 py-5 text-right">
                                    {adv.status !== 'Settled' && (
                                        <button onClick={() => openReturnModal(adv)} className="px-4 py-2 bg-emerald-600 text-white rounded-xl text-[9px] font-black uppercase tracking-widest hover:bg-emerald-700 transition-all shadow-md flex items-center gap-2 ml-auto">
                                            <ArrowDownLeft size={12}/> Return
                                        </button>
                                    )}
                                </td>
                            </tr>
                        );
                    })}
                    {advances.length === 0 && (
                        <tr><td colSpan={5} className="p-12 text-center text-slate-400 text-xs font-bold uppercase italic">No active business advances</td></tr>
                    )}
                    </tbody>
                </table>
            </div>

            {/* LEND MODAL */}
            {isLendModalOpen && (
                <div className="fixed inset-0 bg-slate-900/80 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex justify-between items-center border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-indigo-600 rounded-2xl flex items-center justify-center text-white"><ArrowUpRight size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Issue Advance</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1">Lend Capital to Business</p>
                                </div>
                            </div>
                            <button onClick={() => setIsLendModalOpen(false)} className="p-2 hover:bg-slate-50 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="space-y-5">
                            <IntegratedInput label="Business Partner Name" value={lendForm.partner_name} onChange={(e: any) => setLendForm({...lendForm, partner_name: e.target.value})} icon={User} placeholder="Who is borrowing?" />
                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Amount (PKR)" value={lendForm.amount} onChange={(e: any) => setLendForm({...lendForm, amount: e.target.value})} icon={DollarSign} />
                                <IntegratedInput label="Expected Return (Total)" value={lendForm.expected_return} onChange={(e: any) => setLendForm({...lendForm, expected_return: e.target.value})} icon={PieChart} placeholder="Optional" />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <IntegratedInput label="Issue Date" type="date" value={lendForm.issue_date} onChange={(e: any) => setLendForm({...lendForm, issue_date: e.target.value})} icon={Calendar} />
                                <IntegratedInput label="Due Date" type="date" value={lendForm.due_date} onChange={(e: any) => setLendForm({...lendForm, due_date: e.target.value})} icon={Clock} />
                            </div>
                            <IntegratedSelect label="Source Account" value={lendForm.account_id} onChange={(e: any) => setLendForm({...lendForm, account_id: e.target.value})} options={accounts} displayKey="name" icon={Landmark} />
                            <IntegratedInput label="Notes" value={lendForm.notes} onChange={(e: any) => setLendForm({...lendForm, notes: e.target.value})} icon={StickyNote} placeholder="Terms, Conditions..." />
                        </div>

                        <button onClick={handleLend} className="w-full bg-indigo-600 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-indigo-700 transition-all flex items-center justify-center gap-2">
                            <CheckCircle2 size={16}/> Confirm Transaction
                        </button>
                    </div>
                </div>
            )}

            {/* RETURN MODAL */}
            {isReturnModalOpen && selectedAdvance && (
                <div className="fixed inset-0 bg-slate-900/80 z-[100] flex items-center justify-center p-4 backdrop-blur-sm overflow-y-auto">
                    <div className="bg-white rounded-[2.5rem] w-full max-w-lg p-10 space-y-8 shadow-2xl border border-white/20 my-auto animate-fade-in relative">
                        <div className="flex justify-between items-center border-b border-slate-100 pb-6">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-emerald-600 rounded-2xl flex items-center justify-center text-white"><ArrowDownLeft size={24} /></div>
                                <div>
                                    <h3 className="font-black uppercase italic text-sm tracking-widest leading-none">Receive Return</h3>
                                    <p className="text-[10px] font-bold text-slate-400 uppercase mt-1">From: {selectedAdvance.partner_name}</p>
                                </div>
                            </div>
                            <button onClick={() => setIsReturnModalOpen(false)} className="p-2 hover:bg-slate-50 rounded-full text-slate-400"><X size={20} /></button>
                        </div>

                        <div className="p-4 bg-slate-50 rounded-2xl border border-slate-100 flex justify-between items-center">
                            <div>
                                <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest">Principal Balance</p>
                                <p className="text-sm font-black font-mono text-slate-900">
                                    {formatCurrency(Math.max(0, Number(selectedAdvance.principal_amount) - Number(selectedAdvance.returned_amount)))}
                                </p>
                            </div>
                            <div className="text-right">
                                <p className="text-[9px] font-bold text-slate-400 uppercase tracking-widest">Total Returned</p>
                                <p className="text-sm font-black font-mono text-emerald-600">{formatCurrency(selectedAdvance.returned_amount)}</p>
                            </div>
                        </div>

                        <div className="space-y-5">
                            <IntegratedInput label="Amount Received Now" value={returnForm.amount} onChange={(e: any) => setReturnForm({...returnForm, amount: e.target.value})} icon={DollarSign} />
                            <IntegratedSelect label="Deposit To Account" value={returnForm.account_id} onChange={(e: any) => setReturnForm({...returnForm, account_id: e.target.value})} options={accounts} displayKey="name" icon={Landmark} />
                            <IntegratedInput label="Receipt Date" type="date" value={returnForm.date} onChange={(e: any) => setReturnForm({...returnForm, date: e.target.value})} icon={Calendar} />

                            <div className="flex items-center gap-3 p-4 border border-slate-100 rounded-xl cursor-pointer hover:bg-slate-50 transition-colors" onClick={() => setReturnForm({...returnForm, is_final_settlement: !returnForm.is_final_settlement})}>
                                <div className={`w-5 h-5 rounded border flex items-center justify-center ${returnForm.is_final_settlement ? 'bg-slate-900 border-slate-900 text-white' : 'border-slate-300'}`}>
                                    {returnForm.is_final_settlement && <CheckCircle2 size={12}/>}
                                </div>
                                <div>
                                    <p className="text-xs font-bold text-slate-900 uppercase">Final Settlement</p>
                                    <p className="text-[8px] font-bold text-slate-400 uppercase">Mark loan as fully closed (Profit will be calculated)</p>
                                </div>
                            </div>
                        </div>

                        <button onClick={handleReturn} className="w-full bg-emerald-600 text-white py-5 rounded-2xl font-black uppercase text-xs tracking-widest shadow-xl hover:bg-emerald-700 transition-all flex items-center justify-center gap-2">
                            <CheckCircle2 size={16}/> Confirm Receipt
                        </button>
                    </div>
                </div>
            )}
        </div>
    );
}
