<?php


use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up()
    {
        // --- 1. CONFIGURATION ---
        Schema::create('showroom_settings', function (Blueprint $table) {
            $table->id();
            $table->string('showroom_name')->default('MotoApp Terminal');
            $table->string('address')->nullable();
            $table->string('phone')->nullable();
            $table->json('hidden_modules')->nullable();
            $table->boolean('disable_editing')->default(false);
            $table->boolean('disable_deleting')->default(false);
            $table->timestamps();
        });

        // --- 2. PEOPLE & ENTITIES ---
        Schema::create('accounts', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('type');
            $table->decimal('balance', 15, 2)->default(0);
            $table->string('status')->default('Active');
            $table->timestamps();
        });

        Schema::create('customers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('father_name')->nullable();
            $table->string('cnic')->nullable();
            $table->string('contact')->nullable();
            $table->string('address')->nullable();
            $table->text('notes')->nullable();
            $table->string('ref_number')->nullable();
            $table->timestamps();
        });

        Schema::create('suppliers', function (Blueprint $table) {
            $table->id();
            $table->string('name');
            $table->string('person')->nullable();
            $table->string('contact')->nullable();
            $table->string('email')->nullable();
            $table->string('address')->nullable();
            $table->string('status')->default('Active');
            $table->string('ref_number')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
        });



        // --- 3. INVENTORY CATALOG & PROCUREMENT ---
        Schema::create('products', function (Blueprint $table) {
            $table->id();
            $table->string('make');
            $table->string('model');
            $table->string('year');
            $table->string('color');
            $table->decimal('price', 15, 2);
            $table->decimal('cost', 15, 2);
            $table->string('ref_number')->nullable();
            $table->timestamps();
        });

        Schema::create('product_attributes', function (Blueprint $table) {
            $table->id();
            $table->string('type');
            $table->string('value');
            $table->timestamps();
            $table->unique(['type', 'value']);
        });

        Schema::create('stock_units', function (Blueprint $table) {
            $table->uuid('id')->primary(); // UUID Primary Key
            $table->foreignId('product_id')->constrained('products');
            $table->foreignId('supplier_id')->nullable()->constrained('suppliers');
            $table->string('engine_number')->unique();
            $table->string('chassis_number')->unique();
            $table->decimal('purchase_price', 15, 2);
            $table->date('entry_date');
            $table->string('status')->default('In Stock');
            $table->string('condition')->default('Brand New');
            $table->string('stock_type')->default('Direct Purchase');
            $table->string('ref_number')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
        });

        // New Purchase Order Tables
        Schema::create('purchase_orders', function (Blueprint $table) {
            $table->uuid('id')->primary(); // UUID Primary Key
            $table->foreignId('supplier_id')->constrained('suppliers');
            $table->date('date');
            $table->date('expected_date')->nullable();
            $table->string('status')->default('Pending'); // Pending, Received
            $table->string('ref_number')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
        });

        Schema::create('purchase_order_items', function (Blueprint $table) {
            $table->id();
            $table->foreignUuid('purchase_order_id')->constrained('purchase_orders')->onDelete('cascade');
            $table->foreignId('product_id')->constrained('products');
            $table->integer('qty');
            $table->decimal('cost', 15, 2); // Unit Cost
            $table->timestamps();
        });

        // --- 4. TERMINAL (SALES) ---
        Schema::create('sales', function (Blueprint $table) {
            $table->uuid('id')->primary();
            $table->foreignId('customer_id')->constrained('customers');
            $table->foreignUuid('stock_unit_id')->constrained('stock_units');
            $table->date('sale_date');
            $table->decimal('cost_price', 15, 2);
            $table->decimal('total_amount', 15, 2);
            $table->decimal('down_payment', 15, 2)->default(0);
            $table->string('payment_type');
            $table->integer('installments_count')->default(0);
            $table->decimal('monthly_installment', 15, 2)->default(0);
            $table->string('status')->default('Active');
            $table->text('notes')->nullable();
            $table->string('ref_number')->nullable();
            $table->timestamps();
        });

        Schema::create('payments', function (Blueprint $table) {
            $table->id();
            // Explicit Foreign Key for UUID
            $table->foreignUuid('sale_id')->constrained('sales')->onDelete('cascade');
            $table->foreignId('account_id')->constrained('accounts');
            $table->decimal('amount', 15, 2);
            $table->date('payment_date');
            $table->string('type');
            $table->string('note')->nullable();
            $table->timestamps();
        });

        Schema::create('bookings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('customer_id')->constrained('customers');
            $table->foreignId('product_id')->nullable()->constrained('products');
            $table->decimal('total', 15, 2)->default(0);
            $table->decimal('advance', 15, 2)->default(0);
            $table->string('status')->default('Pending');
            $table->date('booking_date');
            $table->date('promised_date')->nullable();
            $table->text('notes')->nullable();
            $table->string('ref_number')->nullable();
            $table->timestamps();
        });

        // --- 5. FINANCE ---
        Schema::create('expenses', function (Blueprint $table) {
            $table->id();
            $table->string('description');
            $table->decimal('amount', 15, 2);
            $table->string('category')->default('General');
            $table->date('expense_date');
            $table->foreignId('account_id')->constrained('accounts');
            $table->string('ref_number')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
        });

        Schema::create('capital_entries', function (Blueprint $table) {
            $table->id();
            $table->string('description');
            $table->decimal('amount', 15, 2);
            $table->string('type');
            $table->date('entry_date');
            $table->foreignId('account_id')->constrained('accounts');
            $table->string('ref_number')->nullable();
            $table->timestamps();
        });

        Schema::create('fee_batches', function (Blueprint $table) {
            $table->id();
            $table->foreignId('supplier_id')->constrained('suppliers');
            $table->integer('qty');
            $table->decimal('rate', 15, 2);
            $table->decimal('expected_total', 15, 2);
            $table->decimal('received_amount', 15, 2)->default(0);
            $table->date('dispatch_date');
            $table->timestamps();
        });

        // --- 6. OPERATIONS ---


        Schema::create('transfers', function (Blueprint $table) {
            $table->id();
            $table->string('source')->nullable();
            $table->string('destination')->nullable();
            $table->string('engine');
            $table->string('chassis');
            $table->string('type');
            $table->date('transfer_date');
            $table->string('ref_number')->nullable();
            $table->text('notes')->nullable();
            $table->timestamps();
        });

        Schema::create('dealer_exchanges', function (Blueprint $table) {
            $table->id();
            $table->string('dealer_name');
            $table->string('type');
            $table->string('engine_number');
            $table->decimal('exchange_value', 15, 2);
            $table->string('status')->default('Pending');
            $table->timestamps();
        });
    }
};
