<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{Workshop, Claim, Account, Expense, Payment, Supplier};

class ServiceController extends Controller
{
    // --- WORKSHOP MANAGEMENT ---

    public function storeWorkshop(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'contactPerson' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:50',
            'address' => 'nullable|string|max:500',
        ]);

        Workshop::create([
            'name' => $validated['name'],
            'contact_person' => $validated['contactPerson'] ?? null,
            'phone' => $validated['phone'] ?? null,
            'address' => $validated['address'] ?? null,
            'status' => 'Active'
        ]);

        return back()->with('message', 'Workshop Added Directory');
    }

    public function updateWorkshop(Request $request, $id)
    {
        $workshop = Workshop::findOrFail($id);
        $workshop->update($request->all());
        return back()->with('message', 'Workshop Profile Updated');
    }

    // --- CLAIM OPERATIONS ---

    /**
     * Register Claim (The Accrual Step)
     * This creates the Payable (to Workshop) and Receivable (from Supplier).
     */
    public function storeClaim(Request $request)
    {
        $request->validate([
            'type' => 'required',
            'customerName' => 'required_without:saleId',
            'engineNumber' => 'required_without:saleId',
            'workshopId' => 'nullable|exists:workshops,id',
            'supplierId' => 'nullable|exists:suppliers,id'
        ]);

        DB::transaction(function () use ($request) {
            $claim = Claim::create([
                'type' => $request->type,
                'sale_id' => $request->saleId,
                'customer_name' => $request->customerName,
                'engine_number' => $request->engineNumber,
                'description' => $request->description,
                'workshop_cost' => $request->workshopCost ?? 0,
                'claimed_amount' => $request->claimedAmount ?? 0,
                'workshop_id' => $request->workshopId,
                'supplier_id' => $request->supplierId,
                'claim_date' => $request->date ?? now(),
                'status' => 'Pending',
                'workshop_status' => 'Pending',
                'supplier_status' => 'Pending'
            ]);

            // Update Ledgers (Balances)
            // 1. Increase Payable to Workshop
            if ($request->workshopId && $request->workshopCost > 0) {
                Workshop::find($request->workshopId)->increment('balance', $request->workshopCost);
            }

            // 2. Increase Receivable from Supplier (Optional: If you have a balance col on Suppliers)
            // Supplier::find($request->supplierId)->increment('receivable_balance', $request->claimedAmount);
        });

        return back()->with('message', 'Claim Registered & Ledgers Updated');
    }

    /**
     * Step 1: Pay the Workshop (Liability Settlement)
     * This acts as a Vendor Payment, NOT an Expense.
     */
    public function payWorkshop(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id'
        ]);

        DB::transaction(function () use ($request, $id) {
            $claim = Claim::findOrFail($id);

            // 1. Create Vendor Payment (Outflow)
            // We use the vendor_payments table but link it to the workshop
            DB::table('vendor_payments')->insert([
                'workshop_id' => $claim->workshop_id,
                'account_id' => $request->accountId,
                'amount' => $request->amount,
                'payment_date' => now(),
                'notes' => "Claim Payout: {$claim->engine_number} (Claim #{$id})",
                'ref_number' => "CLM-PAY-{$id}",
                'created_at' => now(),
                'updated_at' => now()
            ]);

            // 2. Reduce Cash in Account
            Account::find($request->accountId)->decrement('balance', $request->amount);

            // 3. Reduce Payable to Workshop (Ledger Update)
            if ($claim->workshop_id) {
                Workshop::find($claim->workshop_id)->decrement('balance', $request->amount);
            }

            // 4. Update Claim Status
            $claim->update([
                'workshop_cost' => $request->amount, // Update to actual paid if different
                'workshop_status' => 'Paid'
            ]);
        });

        return back()->with('message', 'Workshop Payment Recorded');
    }

    /**
     * Step 2: Recover from Supplier (Asset Settlement)
     * This acts as Income/Recovery.
     */
    public function recoverFromSupplier(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id'
        ]);

        DB::transaction(function () use ($request, $id) {
            $claim = Claim::findOrFail($id);

            // 1. Record Income (Asset In)
            Payment::create([
                'sale_id' => null, // Not a direct sale invoice
                'account_id' => $request->accountId,
                'amount' => $request->amount,
                'payment_date' => now(),
                'type' => 'Warranty Recovery', // Specific type for reporting
                'note' => "Recovery from {$claim->supplier->name} for Claim #{$id}"
            ]);

            // 2. Increase Cash in Account
            Account::find($request->accountId)->increment('balance', $request->amount);

            // 3. Update Claim Status
            $claim->update([
                'claimed_amount' => $request->amount, // Update to actual received
                'supplier_status' => 'Recovered',
                'status' => 'Completed'
            ]);
        });

        return back()->with('message', 'Funds Recovered from Supplier');
    }
}
