<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Models\{Registration, Account, Payment, Expense};

class RegistrationController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'customerName' => 'required',
            'engineNumber' => 'required',
            'customerFee' => 'nullable|numeric',
            'agentFee' => 'nullable|numeric',
            'accountId' => 'nullable|exists:accounts,id'
        ]);

        DB::transaction(function () use ($request) {
            $customerFee = $request->customerFee ?? 0;
            $agentFee = $request->agentFee ?? 0;
            $profit = $customerFee - $agentFee;

            // Update or Create Registration Record
            $reg = Registration::updateOrCreate(
                ['id' => $request->id ?? (string) Str::uuid()],
                [
                    'type' => $request->type,
                    'sale_id' => $request->saleId,
                    'customer_name' => $request->customerName,
                    'bike_info' => $request->bikeInfo,
                    'engine_number' => $request->engineNumber,
                    'reg_number' => $request->regNumber,
                    'agent_name' => $request->agentName,
                    'customer_fee' => $customerFee,
                    'agent_fee' => $agentFee,
                    'profit' => $profit,
                    'slip_received' => $request->slipReceived ?? false,
                    'doc_received' => $request->docReceived ?? false,
                    'doc_issued' => $request->docIssued ?? false,
                    'registration_date' => $request->registrationDate,
                    'status' => $request->status,
                    'notes' => $request->notes,
                    'ref_number' => $request->refNumber
                ]
            );

            // Handle Financials only if this is a NEW registration to avoid duplicates on edit.
            // If creating a new record AND an account is selected, log income and expense.
            if (!$request->id && $request->accountId) {

                // 1. Income from Customer (Revenue)
                if ($customerFee > 0) {
                    Payment::create([
                        'account_id' => $request->accountId,
                        'amount' => $customerFee,
                        'payment_date' => $request->registrationDate,
                        'type' => 'Registration Income', // Custom type for filtering
                        'note' => "Reg Fee Received: {$request->engineNumber} ({$request->customerName})"
                    ]);
                    Account::find($request->accountId)->increment('balance', $customerFee);
                }

                // 2. Expense to Agent (Cost)
                if ($agentFee > 0) {
                    Expense::create([
                        'account_id' => $request->accountId,
                        'amount' => $agentFee,
                        'category' => 'Registration',
                        'expense_date' => $request->registrationDate,
                        'description' => "Agent Fee Paid: {$request->engineNumber}",
                        'notes' => "Agent: {$request->agentName}",
                        'ref_number' => $request->refNumber
                    ]);
                    Account::find($request->accountId)->decrement('balance', $agentFee);
                }
            }
        });

        return back()->with('message', 'Registration Record Processed');
    }
}
