<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use App\Models\{Product, ProductAttribute, StockUnit, PurchaseOrder, PurchaseOrderItem, Account, Supplier};

class InventoryController extends Controller
{
    public function storeProduct(Request $request) {
        $data = $request->validate([
            'make' => 'required', 'model' => 'required', 'price' => 'required',
            'year' => 'nullable', 'color' => 'nullable', 'cost' => 'nullable'
        ]);

        Product::updateOrCreate(
            ['id' => $request->id],
            $data
        );
        return back()->with('message', 'Product Saved');
    }

    public function storeAttribute(Request $request) {
        ProductAttribute::firstOrCreate([
            'type' => $request->type,
            'value' => $request->value
        ]);
        return back();
    }

    /**
     * Store Manual Stock (Direct Purchase without PO)
     */
    public function storeStock(Request $request) {
        $request->validate([
            'productId' => 'required',
            'engineNumber' => 'required|unique:stock_units,engine_number',
            'purchasePrice' => 'required|numeric',
            'date' => 'required|date'
        ]);

        DB::transaction(function() use ($request) {
            StockUnit::create([
                'id' => (string) Str::uuid(),
                'product_id' => $request->productId,
                'engine_number' => $request->engineNumber,
                'chassis_number' => $request->chassisNumber,
                'purchase_price' => $request->purchasePrice,
                'supplier_id' => $request->supplierId,
                'entry_date' => $request->date,
                'ref_number' => $request->refNumber,
                'condition' => $request->condition ?? 'Brand New',
                'stock_type' => 'Direct Purchase',
                'notes' => $request->notes,
                'status' => 'In Stock'
            ]);

            // Handle Direct Payment
            if ($request->paidAmount > 0 && $request->accountId) {
                DB::table('vendor_payments')->insert([
                    'supplier_id' => $request->supplierId,
                    'account_id' => $request->accountId,
                    'amount' => $request->paidAmount,
                    'payment_date' => $request->date,
                    'notes' => 'Direct Stock Purchase: ' . $request->engineNumber,
                    'ref_number' => $request->refNumber,
                    'created_at' => now(), 'updated_at' => now()
                ]);
                Account::find($request->accountId)->decrement('balance', $request->paidAmount);
            }
        });

        return back()->with('message', 'Direct Stock Added');
    }

    /**
     * Create Purchase Order
     */
    public function storePurchaseOrder(Request $request) {
        $request->validate([
            'supplierId' => 'required',
            'items' => 'required|array|min:1'
        ]);

        DB::transaction(function () use ($request) {
            $poId = (string) Str::uuid();
            $po = PurchaseOrder::create([
                'id' => $poId,
                'supplier_id' => $request->supplierId,
                'date' => $request->date,
                'expected_date' => $request->expectedDate,
                'ref_number' => $request->refNumber,
                'notes' => $request->notes,
                'status' => 'Pending'
            ]);

            foreach ($request->items as $item) {
                PurchaseOrderItem::create([
                    'purchase_order_id' => $poId,
                    'product_id' => $item['productId'],
                    'qty' => $item['qty'],
                    'cost' => $item['cost'],
                    'received_qty' => 0
                ]);
            }

            // Handle Advance Payment
            if ($request->paidAmount > 0 && $request->accountId) {
                DB::table('vendor_payments')->insert([
                    'supplier_id' => $request->supplierId,
                    'account_id' => $request->accountId,
                    'amount' => $request->paidAmount,
                    'payment_date' => $request->date,
                    'notes' => 'Advance for PO ' . $request->refNumber,
                    'created_at' => now(), 'updated_at' => now()
                ]);
                Account::find($request->accountId)->decrement('balance', $request->paidAmount);
            }
        });
        return back()->with('message', 'Purchase Order Created');
    }
    /**
     * Modify Stock
     */

    public function modifyStock(Request $request, $id)
    {
        $request->validate([
            'partDescription' => 'required|string',
            'costAmount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id',
            'date' => 'required|date'
        ]);

        \Illuminate\Support\Facades\DB::transaction(function () use ($request, $id) {
            $unit = \App\Models\StockUnit::findOrFail($id);

            // 1. Record the Modification Log
            \App\Models\StockModification::create([
                'stock_unit_id' => $unit->id,
                'part_description' => $request->partDescription,
                'cost_amount' => $request->costAmount,
                'account_id' => $request->accountId,
                'date' => $request->date,
                'ref_number' => $request->refNumber
            ]);

            // 2. Capitalize the Asset (Increase Stock Value)
            $unit->increment('purchase_price', $request->costAmount);

            // 3. Record the Cash Outflow (Expense/Payment)
            // We log this as an 'Expense' but categorize it as 'Asset Capitalization'
            // OR as a direct Payment if you prefer payment ledger.
            // Using 'vendor_payments' table might be tricky if no specific supplier is involved.
            // Best approach: Use Expenses table or generic ledger entry.

            \App\Models\Expense::create([
                'account_id' => $request->accountId,
                'amount' => $request->costAmount,
                'category' => 'Stock Enhancement', // Special category
                'description' => "Part for {$unit->engine_number}: {$request->partDescription}",
                'expense_date' => $request->date,
                'ref_number' => $request->refNumber
            ]);

            // 4. Update Account Balance
            \App\Models\Account::find($request->accountId)->decrement('balance', $request->costAmount);
        });

        return back()->with('message', 'Stock Enhanced & Cost Recorded');
    }




    /**
     * Receive Items against a Purchase Order
     * This converts 'Paper Orders' into 'Physical Stock'
     */
    public function receivePurchaseOrder(Request $request, $id) {
        $request->validate([
            'items' => 'required|array',
            'items.*.engine' => 'required|unique:stock_units,engine_number',
            'items.*.poItemId' => 'required|exists:purchase_order_items,id',
            'date' => 'required|date',
        ]);

        DB::transaction(function () use ($request, $id) {
            $po = PurchaseOrder::findOrFail($id);

            // 1. Process each received item line
            foreach ($request->items as $receivedItem) {
                $poItem = PurchaseOrderItem::findOrFail($receivedItem['poItemId']);

                // Create the Physical Stock Unit
                StockUnit::create([
                    'id' => (string) Str::uuid(),
                    'product_id' => $poItem->product_id,
                    'supplier_id' => $po->supplier_id,
                    'engine_number' => $receivedItem['engine'],
                    'chassis_number' => $receivedItem['chassis'] ?? null,
                    'purchase_price' => $poItem->cost, // Cost inherited from PO agreement
                    'entry_date' => $request->date,
                    'status' => 'In Stock',
                    'condition' => 'Brand New',
                    'stock_type' => 'PO Receipt',
                    'ref_number' => $request->refNumber,
                    'notes' => 'Received against PO ' . ($po->ref_number ?? $po->id)
                ]);

                // Update PO Line Item Progress
                $poItem->increment('received_qty');
            }

            // 2. Update Master PO Status
            // We check if all items in the PO have been fully received
            $po->refresh();
            $allItemsReceived = $po->items->every(function($item) {
                return $item->received_qty >= $item->qty;
            });

            $po->update(['status' => $allItemsReceived ? 'Received' : 'Partial']);

            // 3. Handle Payment at time of Receipt (if any)
            if ($request->paidAmount > 0 && $request->accountId) {
                DB::table('vendor_payments')->insert([
                    'supplier_id' => $po->supplier_id,
                    'account_id' => $request->accountId,
                    'amount' => $request->paidAmount,
                    'payment_date' => $request->date,
                    'notes' => 'Payment on Receipt PO-' . substr($po->id, 0, 6),
                    'ref_number' => $request->refNumber,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                Account::find($request->accountId)->decrement('balance', $request->paidAmount);
            }
        });

        return back()->with('message', 'Stock Units Received & Added to Inventory');
    }
}
