<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{Account, Expense, CapitalEntry, FeeBatch};

class FinanceController extends Controller {

    public function storeExpense(Request $request) {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id',
            'description' => 'required'
        ]);

        DB::transaction(function() use ($request) {
            Expense::create([
                'description' => $request->description,
                'amount' => $request->amount,
                'category' => $request->category ?? 'General',
                'expense_date' => $request->date,
                'account_id' => $request->accountId,
                'notes' => $request->notes,
                'ref_number' => $request->refNumber
            ]);

            Account::where('id', $request->accountId)->decrement('balance', $request->amount);
        });

        return back()->with('message', 'Expense Recorded');
    }

    public function storeCapital(Request $request) {
        $request->validate([
            'amount' => 'required|numeric|min:0',
            'accountId' => 'required|exists:accounts,id',
            'type' => 'required|in:Investment,Withdrawal'
        ]);

        DB::transaction(function() use ($request) {
            CapitalEntry::create([
                'description' => $request->description,
                'amount' => $request->amount,
                'type' => $request->type,
                'entry_date' => $request->date,
                'account_id' => $request->accountId,
                'ref_number' => $request->refNumber
            ]);

            $account = Account::where('id', $request->accountId);
            if ($request->type === 'Investment') {
                $account->increment('balance', $request->amount);
            } else {
                $account->decrement('balance', $request->amount);
            }
        });

        return back()->with('message', 'Capital Ledger Updated');
    }

    public function storeAccount(Request $request) {
        Account::create($request->all());
        return back()->with('message', 'Account Created');
    }

    public function transferFunds(Request $request) {
        $request->validate([
            'fromAccountId' => 'required|exists:accounts,id',
            'toAccountId' => 'required|exists:accounts,id|different:fromAccountId',
            'amount' => 'required|numeric|min:0'
        ]);

        DB::transaction(function() use ($request) {
            // Check balance
            $fromAcc = Account::lockForUpdate()->find($request->fromAccountId);
            if($fromAcc->balance < $request->amount) {
                throw new \Exception("Insufficient funds in source account");
            }

            $fromAcc->decrement('balance', $request->amount);
            Account::where('id', $request->toAccountId)->increment('balance', $request->amount);
        });

        return back()->with('message', 'Funds Transferred');
    }

    public function dispatchFeeBatch(Request $request) {
        FeeBatch::create([
            'supplier_id' => $request->supplierId,
            'qty' => $request->qty,
            'rate' => $request->rate,
            'expected_total' => $request->qty * $request->rate,
            'received_amount' => 0, // Initially 0 until claimed/processed
            'dispatch_date' => $request->date
        ]);
        return back()->with('message', 'Fee Batch Dispatched');
    }
}
